import React from 'react';
import { Document, StyleSheet, View } from '@react-pdf/renderer';
import { capitalize } from 'lodash';
import { INVOICE_INSURER_PAYMENT_STATUSES } from '@tamanu/constants';
import { CertificateHeader, Watermark } from './Layout';
import { LetterheadSection } from './LetterheadSection';
import { MultiPageHeader } from './printComponents/MultiPageHeader';
import { getName } from '../patientAccessors';
import { Footer } from './printComponents/Footer';
import { formatShort } from '@tamanu/utils/dateTime';
import { InvoiceDetails } from './printComponents/InvoiceDetails';
import { getInsurerDiscountAmountDisplayList, getInvoiceItemDiscountPriceDisplay, getInvoiceItemPriceDisplay, getInvoiceSummaryDisplay, getPatientPaymentsWithRemainingBalanceDisplay, formatDisplayPrice, getInsurerPaymentsWithRemainingBalanceDisplay } from '../invoice';
import { withLanguageContext } from '../pdf/languageContext';
import { Page } from '../pdf/Page';
import { Text } from '../pdf/Text';
import { PatientDetails } from './printComponents/PatientDetails';
import { InvoiceEncounterDetails } from './printComponents/InvoiceEncounterDetails';
const borderStyle = '1 solid black';
const pageStyles = StyleSheet.create({
    body: {
        paddingHorizontal: 50,
        paddingTop: 30,
        paddingBottom: 50
    }
});
const textStyles = StyleSheet.create({
    sectionTitle: {
        marginBottom: 3,
        fontSize: 11
    }
});
const tableStyles = StyleSheet.create({
    table: {
        flexDirection: 'column',
        marginBottom: 5
    },
    row: {
        flexDirection: 'row',
        justifyContent: 'space-evenly',
        borderTop: borderStyle,
        borderRight: borderStyle,
        borderBottom: borderStyle,
        marginBottom: -1
    },
    baseCell: {
        flexDirection: 'row',
        borderLeft: borderStyle,
        alignItems: 'flex-start',
        padding: 7
    },
    p: {
        fontSize: 9
    },
    noteText: {
        fontSize: 7,
        marginTop: 1
    }
});
const priceCellStyles = StyleSheet.create({
    container: {
        justifyContent: 'space-between',
        width: '100%',
        flexDirection: 'row'
    },
    crossOutText: {
        textDecoration: 'line-through'
    }
});
const summaryPaneStyles = StyleSheet.create({
    container: {
        width: 220,
        marginLeft: 'auto',
        paddingHorizontal: 7,
        paddingVertical: 3,
        border: borderStyle
    },
    item: {
        flexDirection: 'row',
        justifyContent: 'space-between',
        marginTop: 4.5,
        marginBottom: 4.5
    },
    subItem: {
        width: 56,
        flexDirection: 'row',
        justifyContent: 'space-between'
    }
});
const HorizontalRule = ({ width = '1px' })=>{
    return /*#__PURE__*/ React.createElement(View, {
        style: {
            borderBottom: `${width} solid black`
        }
    });
};
const Table = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: tableStyles.table,
        ...props
    });
const Row = (props)=>/*#__PURE__*/ React.createElement(View, {
        style: tableStyles.row,
        ...props
    });
const P = ({ style = {}, children, bold })=>/*#__PURE__*/ React.createElement(Text, {
        bold: bold,
        style: [
            tableStyles.p,
            style
        ]
    }, children);
const Cell = ({ children, style = {} })=>/*#__PURE__*/ React.createElement(View, {
        style: [
            tableStyles.baseCell,
            style
        ]
    }, /*#__PURE__*/ React.createElement(P, null, children));
const CustomCellComponent = ({ children, style = {} })=>/*#__PURE__*/ React.createElement(View, {
        style: [
            tableStyles.baseCell,
            style
        ]
    }, children);
const getPrice = (item)=>{
    const price = getInvoiceItemPriceDisplay(item);
    const discountPrice = getInvoiceItemDiscountPriceDisplay(item);
    return /*#__PURE__*/ React.createElement(View, {
        style: priceCellStyles.container
    }, /*#__PURE__*/ React.createElement(P, {
        style: discountPrice ? priceCellStyles.crossOutText : undefined
    }, price), !!discountPrice && /*#__PURE__*/ React.createElement(P, null, discountPrice));
};
const getInvoiceItemDetails = (item)=>{
    const name = item.productName;
    const note = item.note;
    return /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(P, null, name, !item.productDiscountable && ' (non-discountable)')), !!note && /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(P, {
        style: [
            tableStyles.noteText
        ]
    }, "Note: ", note)));
};
const HeaderCell = ({ children, style })=>/*#__PURE__*/ React.createElement(View, {
        style: [
            tableStyles.baseCell,
            style
        ]
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, children));
const SectionSpacing = ()=>/*#__PURE__*/ React.createElement(View, {
        style: {
            paddingBottom: '10px'
        }
    });
const getInsurerPaymentStatus = (insurerPayment)=>{
    if (insurerPayment?.status === INVOICE_INSURER_PAYMENT_STATUSES.REJECTED) {
        return /*#__PURE__*/ React.createElement(P, null, `${capitalize(insurerPayment?.status)}${insurerPayment?.reason ? ` (${insurerPayment?.reason})` : ''}`);
    }
    return capitalize(insurerPayment?.status);
};
const COLUMNS = {
    invoiceItems: [
        {
            key: 'orderDate',
            title: 'Date',
            style: {
                width: '12%'
            },
            accessor: ({ orderDate })=>orderDate ? formatShort(orderDate) : '--/--/----'
        },
        {
            key: 'productName',
            title: 'Details',
            style: {
                width: '34%'
            },
            accessor: (row)=>getInvoiceItemDetails(row),
            CellComponent: CustomCellComponent
        },
        {
            key: 'productCode',
            title: 'Code',
            style: {
                width: '10%'
            },
            accessor: ({ productCode })=>productCode
        },
        {
            key: 'quantity',
            title: 'Quantity',
            style: {
                width: '11%'
            },
            accessor: ({ quantity })=>quantity
        },
        {
            key: 'orderedBy',
            title: 'Ordered by',
            accessor: ({ orderedByUser })=>orderedByUser?.displayName,
            style: {
                width: '14%'
            }
        },
        {
            key: 'price',
            title: 'Price',
            accessor: (row)=>getPrice(row),
            style: {
                width: '19%'
            },
            CellComponent: CustomCellComponent
        }
    ],
    patientPayments: [
        {
            key: 'date',
            title: 'Date',
            style: {
                width: '15%'
            },
            accessor: ({ date })=>date ? formatShort(date) : '--/--/----'
        },
        {
            key: 'methodName',
            title: 'Method',
            style: {
                width: '28%'
            },
            accessor: ({ patientPayment })=>patientPayment?.method?.name
        },
        {
            key: 'amount',
            title: 'Amount',
            style: {
                width: '15%'
            },
            accessor: ({ amount })=>formatDisplayPrice(amount)
        },
        {
            key: 'receiptNumber',
            title: 'Receipt number',
            accessor: ({ receiptNumber })=>receiptNumber,
            style: {
                width: '21%'
            }
        },
        {
            key: 'remainingBalance',
            title: 'Remaining balance',
            accessor: ({ remainingBalance })=>remainingBalance,
            style: {
                width: '21%'
            }
        }
    ],
    insurerPayments: [
        {
            key: 'date',
            title: 'Date',
            style: {
                width: '15%'
            },
            accessor: ({ date })=>date ? formatShort(date) : '--/--/----'
        },
        {
            key: 'insurerName',
            title: 'Payer',
            style: {
                width: '17%'
            },
            accessor: ({ insurerPayment })=>insurerPayment?.insurer?.name
        },
        {
            key: 'amount',
            title: 'Amount',
            style: {
                width: '11%'
            },
            accessor: ({ amount })=>amount
        },
        {
            key: 'receiptNumber',
            title: 'Receipt number',
            accessor: ({ receiptNumber })=>receiptNumber,
            style: {
                width: '14%'
            }
        },
        {
            key: 'remainingBalance',
            title: 'Remaining balance',
            accessor: ({ remainingBalance })=>remainingBalance,
            style: {
                width: '13%'
            }
        },
        {
            key: 'status',
            title: 'Status',
            accessor: ({ insurerPayment })=>getInsurerPaymentStatus(insurerPayment),
            style: {
                width: '30%'
            }
        }
    ]
};
const MultipageTableHeading = ({ title, style = textStyles.sectionTitle })=>{
    let firstPageOccurrence = Number.MAX_SAFE_INTEGER;
    return /*#__PURE__*/ React.createElement(Text, {
        bold: true,
        fixed: true,
        style: style,
        render: ({ pageNumber, subPageNumber })=>{
            if (pageNumber < firstPageOccurrence && subPageNumber) {
                firstPageOccurrence = pageNumber;
            }
            return pageNumber === firstPageOccurrence ? title : `${title} cont...`;
        }
    });
};
const DataTableHeading = ({ columns, title })=>{
    return /*#__PURE__*/ React.createElement(View, {
        fixed: true
    }, title && /*#__PURE__*/ React.createElement(MultipageTableHeading, {
        title: title
    }), /*#__PURE__*/ React.createElement(Row, {
        wrap: false
    }, columns.map(({ key, title, style })=>/*#__PURE__*/ React.createElement(HeaderCell, {
            key: key,
            style: style
        }, title))));
};
const DataTable = ({ data, columns, title })=>/*#__PURE__*/ React.createElement(Table, null, /*#__PURE__*/ React.createElement(DataTableHeading, {
        columns: columns,
        title: title
    }), data.map((row)=>/*#__PURE__*/ React.createElement(Row, {
            key: row.id,
            wrap: false
        }, columns.map(({ key, accessor, style, CellComponent })=>{
            const displayValue = accessor ? accessor(row) : row[key] || '';
            if (CellComponent) {
                return /*#__PURE__*/ React.createElement(CellComponent, {
                    key: key,
                    style: style
                }, displayValue);
            }
            return /*#__PURE__*/ React.createElement(Cell, {
                key: key,
                style: style
            }, displayValue);
        }))));
const TableSection = ({ title, data, columns, type })=>{
    return /*#__PURE__*/ React.createElement(View, null, /*#__PURE__*/ React.createElement(View, {
        minPresenceAhead: 70
    }), /*#__PURE__*/ React.createElement(DataTable, {
        data: data,
        columns: columns,
        title: title,
        type: type
    }), /*#__PURE__*/ React.createElement(SectionSpacing, null));
};
const SummaryPane = ({ invoice })=>{
    const { discountableItemsSubtotal, nonDiscountableItemsSubtotal, itemsSubtotal, patientSubtotal, patientDiscountableSubtotal, discountTotal, patientTotal } = getInvoiceSummaryDisplay(invoice);
    const insurerDiscountAmountDisplayList = getInsurerDiscountAmountDisplayList(invoice?.insurers, itemsSubtotal);
    return /*#__PURE__*/ React.createElement(View, {
        wrap: false,
        style: summaryPaneStyles.container
    }, /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, null, "Discountable items subtotal"), /*#__PURE__*/ React.createElement(P, null, discountableItemsSubtotal ?? '-')), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, null, "Non-discountable items subtotal"), /*#__PURE__*/ React.createElement(P, null, nonDiscountableItemsSubtotal ?? '-')), /*#__PURE__*/ React.createElement(HorizontalRule, null), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, "Total"), /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, itemsSubtotal ?? '-')), /*#__PURE__*/ React.createElement(HorizontalRule, null), invoice?.insurers?.length && /*#__PURE__*/ React.createElement(React.Fragment, null, /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, "Insurer")), invoice?.insurers.map((insurer, index)=>{
        return /*#__PURE__*/ React.createElement(View, {
            key: insurer.id,
            style: summaryPaneStyles.item
        }, /*#__PURE__*/ React.createElement(P, null, insurer.insurer?.name), /*#__PURE__*/ React.createElement(View, {
            style: summaryPaneStyles.subItem
        }, /*#__PURE__*/ React.createElement(P, null, insurer.percentage * 100, "%"), /*#__PURE__*/ React.createElement(P, null, `-${insurerDiscountAmountDisplayList[index]}`)));
    }), /*#__PURE__*/ React.createElement(HorizontalRule, null)), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, "Patient subtotal"), /*#__PURE__*/ React.createElement(P, null, patientSubtotal)), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, "Discount"), !!invoice.discount && /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.subItem
    }, /*#__PURE__*/ React.createElement(P, null, invoice.discount?.percentage * 100, "%"), /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, typeof discountTotal === 'string' ? `-${discountTotal}` : '-'))), !!invoice.discount && /*#__PURE__*/ React.createElement(React.Fragment, null, /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, invoice.discount?.isManual ? /*#__PURE__*/ React.createElement(P, null, "Manual discount") : /*#__PURE__*/ React.createElement(P, null, "Patient discount applied")), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.item
    }, /*#__PURE__*/ React.createElement(P, null, "Applied to discountable balance"), /*#__PURE__*/ React.createElement(View, {
        style: summaryPaneStyles.subItem
    }, /*#__PURE__*/ React.createElement(P, null, patientDiscountableSubtotal ?? '-')))), /*#__PURE__*/ React.createElement(HorizontalRule, null), /*#__PURE__*/ React.createElement(View, {
        style: [
            summaryPaneStyles.item,
            {
                marginVertical: 7.5
            }
        ]
    }, /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, "Patient total"), /*#__PURE__*/ React.createElement(P, {
        bold: true
    }, patientTotal ?? '-')));
};
const InvoiceRecordPrintoutComponent = ({ patientData, encounter, certificateData, discharge, getLocalisation, getSetting, clinicianText, invoice, enablePatientInsurer })=>{
    const { watermark, logo } = certificateData;
    const patientPayments = getPatientPaymentsWithRemainingBalanceDisplay(invoice);
    const insurerPayments = getInsurerPaymentsWithRemainingBalanceDisplay(invoice);
    return /*#__PURE__*/ React.createElement(Document, null, /*#__PURE__*/ React.createElement(Page, {
        size: "A4",
        style: pageStyles.body,
        wrap: true
    }, watermark && /*#__PURE__*/ React.createElement(Watermark, {
        src: watermark
    }), /*#__PURE__*/ React.createElement(MultiPageHeader, {
        documentName: `Invoice number: ${invoice.displayId}`,
        patientId: patientData.displayId,
        patientName: getName(patientData)
    }), /*#__PURE__*/ React.createElement(CertificateHeader, null, /*#__PURE__*/ React.createElement(LetterheadSection, {
        logoSrc: logo,
        certificateTitle: `Invoice number: ${invoice.displayId}`,
        letterheadConfig: certificateData
    })), /*#__PURE__*/ React.createElement(SectionSpacing, null), /*#__PURE__*/ React.createElement(PatientDetails, {
        getLocalisation: getLocalisation,
        getSetting: getSetting,
        patient: patientData
    }), /*#__PURE__*/ React.createElement(SectionSpacing, null), /*#__PURE__*/ React.createElement(InvoiceEncounterDetails, {
        encounter: encounter,
        discharge: discharge,
        clinicianText: clinicianText
    }), /*#__PURE__*/ React.createElement(SectionSpacing, null), /*#__PURE__*/ React.createElement(InvoiceDetails, {
        encounter: encounter,
        invoice: invoice,
        patient: patientData,
        enablePatientInsurer: enablePatientInsurer
    }), /*#__PURE__*/ React.createElement(SectionSpacing, null), invoice?.items?.length > 0 && /*#__PURE__*/ React.createElement(TableSection, {
        data: invoice?.items,
        columns: COLUMNS.invoiceItems
    }), /*#__PURE__*/ React.createElement(SummaryPane, {
        invoice: invoice
    }), /*#__PURE__*/ React.createElement(SectionSpacing, null), patientPayments?.length && /*#__PURE__*/ React.createElement(TableSection, {
        title: "Patient payment",
        data: patientPayments,
        columns: COLUMNS.patientPayments
    }), insurerPayments?.length && /*#__PURE__*/ React.createElement(TableSection, {
        title: "Insurer payment",
        data: insurerPayments,
        columns: COLUMNS.insurerPayments
    }), /*#__PURE__*/ React.createElement(Footer, null)));
};
export const InvoiceRecordPrintout = withLanguageContext(InvoiceRecordPrintoutComponent);

//# sourceMappingURL=InvoiceRecordPrintout.js.map