"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    insurerPaymentImporter: function() {
        return insurerPaymentImporter;
    },
    parseExcel: function() {
        return parseExcel;
    }
});
const _xlsx = /*#__PURE__*/ _interop_require_wildcard(require("xlsx"));
const _lodash = require("lodash");
const _zod = require("zod");
const _nanoid = require("nanoid");
const _invoice = require("@tamanu/shared/utils/invoice");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _decimal = /*#__PURE__*/ _interop_require_default(require("decimal.js"));
const _stats = require("../stats");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
const parseExcel = (filePath)=>{
    const workbook = _xlsx.readFile(filePath);
    return (0, _lodash.chain)(workbook.Sheets).mapValues((sheet)=>_xlsx.utils.sheet_to_json(sheet)).value();
};
//* The random generator should use a format of 8 alphanumeric characters but not include 0, O or I - please check this is sufficient to guarantee uniqueness with 100 entries/day. Letters should be capitalised.
const receiptNumberGenerator = (0, _nanoid.customAlphabet)('123456789ABCDEFGHJKLMNPQRSTWUVXYZ', 8);
const insurerPaymentImportSchema = _zod.z.object({
    id: _zod.z.string().uuid(),
    date: _zod.z.string().date(),
    amount: _zod.z.coerce.number().min(0).transform((amount)=>(0, _invoice.round)(amount, 2)),
    invoiceNumber: _zod.z.string(),
    insurerId: _zod.z.string(),
    reason: _zod.z.string().optional()
}).strip().transform((data)=>({
        ...data,
        receiptNumber: receiptNumberGenerator()
    }));
async function insurerPaymentImporter({ errors, models, stats, file, checkPermission }) {
    checkPermission('create', 'InvoicePayment');
    const workbook = parseExcel(file);
    const sheet = Object.values(workbook).at(0);
    const sheetName = Object.keys(workbook).at(0);
    const subStat = {};
    if (!sheet) {
        throw new Error('No sheet found in workbook');
    }
    let index = 0;
    for await (const row of sheet){
        const { data, error } = await insurerPaymentImportSchema.safeParseAsync(row);
        if (error) {
            errors.push(new _errors.ValidationError(sheetName, index, error));
            continue;
        }
        const countInvoices = await models.Invoice.count({
            where: {
                displayId: data.invoiceNumber
            }
        });
        if (!countInvoices) {
            errors.push(new _errors.ValidationError(sheetName, index, 'Invoice not found'));
            continue;
        }
        if (countInvoices > 1) {
            errors.push(new _errors.ValidationError(sheetName, index, 'Multiple invoices found'));
            continue;
        }
        const invoice = await models.Invoice.findOne({
            where: {
                displayId: data.invoiceNumber
            },
            include: models.Invoice.getFullReferenceAssociations()
        });
        if (!invoice) {
            errors.push(new _errors.ValidationError(sheetName, index, 'Invoice not found'));
            continue;
        }
        if (invoice.status !== _constants.INVOICE_STATUSES.FINALISED) {
            errors.push(new _errors.ValidationError(sheetName, index, 'Invoice is not finalised'));
            continue;
        }
        const { itemsSubtotal, insurerPaymentsTotal: allInsurerPaymentsTotal, insurerDiscountTotal: allInsurerDiscountTotal } = (0, _invoice.getInvoiceSummary)(invoice);
        const { insurerDiscountTotal, insurerPaymentRemainingBalance } = (0, _invoice.getSpecificInsurerPaymentRemainingBalance)(invoice?.insurers ?? [], invoice?.payments ?? [], data.insurerId, itemsSubtotal);
        try {
            //check if the insurer payment already exists
            const insurerPayment = await models.InvoiceInsurerPayment.findByPk(data.id, {
                include: models.InvoiceInsurerPayment.getFullReferenceAssociations()
            });
            if (insurerPayment) {
                checkPermission('write', 'InvoicePayment');
                //update the payment
                if (data.amount > (0, _invoice.round)(new _decimal.default(insurerPaymentRemainingBalance).add(insurerPayment.detail.amount).toNumber(), 2)) {
                    errors.push(new _errors.ValidationError(sheetName, index, 'Amount is greater than the amount owing'));
                    continue;
                }
                await models.InvoicePayment.update({
                    invoiceId: invoice.id,
                    date: data.date,
                    amount: data.amount
                }, {
                    where: {
                        id: insurerPayment.invoicePaymentId
                    }
                });
                await models.InvoiceInsurerPayment.update({
                    insurerId: data.insurerId,
                    reason: data.reason,
                    status: data.amount === 0 ? _constants.INVOICE_INSURER_PAYMENT_STATUSES.REJECTED : data.amount === (0, _invoice.round)(insurerDiscountTotal, 2) ? _constants.INVOICE_INSURER_PAYMENT_STATUSES.PAID : _constants.INVOICE_INSURER_PAYMENT_STATUSES.PARTIAL
                }, {
                    where: {
                        id: insurerPayment.id
                    }
                });
                //Update the overall insurer payment status to invoice
                await models.Invoice.update({
                    insurerPaymentStatus: (0, _invoice.getInvoiceInsurerPaymentStatus)(new _decimal.default(allInsurerPaymentsTotal).minus(insurerPayment.detail.amount).add(data.amount).toNumber(), allInsurerDiscountTotal)
                }, {
                    where: {
                        id: invoice.id
                    }
                });
                (0, _stats.updateStat)(subStat, (0, _stats.statkey)('InvoiceInsurerPayment', sheetName), 'updated');
            } else {
                //create new payment
                //* Block payment if the amount is greater than the amount owing of this insurer
                if (data.amount > (0, _invoice.round)(insurerPaymentRemainingBalance, 2)) {
                    errors.push(new _errors.ValidationError(sheetName, index, 'Amount is greater than the amount owing'));
                    continue;
                }
                const payment = await models.InvoicePayment.create({
                    invoiceId: invoice.id,
                    date: data.date,
                    receiptNumber: data.receiptNumber,
                    amount: data.amount
                }, {
                    returning: true
                });
                await models.InvoiceInsurerPayment.create({
                    id: data.id,
                    invoicePaymentId: payment.id,
                    insurerId: data.insurerId,
                    reason: data.reason,
                    status: data.amount === 0 ? _constants.INVOICE_INSURER_PAYMENT_STATUSES.REJECTED : data.amount === (0, _invoice.round)(insurerDiscountTotal, 2) ? _constants.INVOICE_INSURER_PAYMENT_STATUSES.PAID : _constants.INVOICE_INSURER_PAYMENT_STATUSES.PARTIAL
                });
                //Update the overall insurer payment status to invoice
                await models.Invoice.update({
                    insurerPaymentStatus: (0, _invoice.getInvoiceInsurerPaymentStatus)(new _decimal.default(allInsurerPaymentsTotal).add(data.amount).toNumber(), allInsurerDiscountTotal)
                }, {
                    where: {
                        id: invoice.id
                    }
                });
                (0, _stats.updateStat)(subStat, (0, _stats.statkey)('InvoiceInsurerPayment', sheetName), 'created');
            }
        } catch (e) {
            errors.push(new _errors.ValidationError('', index, e));
        }
        index++;
    }
    stats.push(subStat);
}

//# sourceMappingURL=insurerPaymentImporter.js.map