"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    createVdsNcTestData: function() {
        return createVdsNcTestData;
    },
    createVdsNcVaccinationData: function() {
        return createVdsNcVaccinationData;
    }
});
const _datefnstz = require("date-fns-tz");
const _transliteration = require("transliteration");
const _logging = require("@tamanu/shared/services/logging");
const _localisation = require("../../localisation");
const SEX_TO_CHAR = {
    male: 'M',
    female: 'F',
    other: 'O'
};
const SCHEDULE_TO_SEQUENCE = {
    'Dose 1': 1,
    'Dose 2': 2,
    Booster: 3,
    'Dose 3': 3,
    'Dose 4': 4,
    'Dose 5': 5,
    'Dose 6': 6,
    'Dose 7': 7,
    'Dose 8': 8,
    'Dose 9': 9
};
const METHOD_CODE = {
    GeneXpert: 'antigen',
    RTPCR: 'molecular(PCR)',
    RDT: 'antigen'
};
const DATE_FORMAT_ISODATE = 'yyyy-MM-dd';
const DATE_FORMAT_RFC3339 = "yyyy-MM-dd'T'HH:mm:ssxxx";
const createVdsNcVaccinationData = async (patientId, { models })=>{
    const { Patient, PatientAdditionalData, ReferenceData, Encounter, Facility, Location, ScheduledVaccine, CertifiableVaccine } = models;
    const { country, timeZone } = await (0, _localisation.getLocalisation)();
    const countryCode = country['alpha-3'];
    const patient = await Patient.findOne({
        where: {
            id: patientId
        }
    });
    const { firstName, lastName, dateOfBirth, sex } = patient;
    const pad = await PatientAdditionalData.findOne({
        where: {
            patientId
        }
    });
    const passport = pad?.passport;
    const { data: vaccinations } = await patient.getAdministeredVaccines({
        order: [
            [
                'date',
                'ASC'
            ]
        ],
        include: [
            {
                model: Location,
                as: 'location',
                include: [
                    {
                        model: Facility,
                        as: 'facility'
                    }
                ]
            },
            {
                model: Encounter,
                as: 'encounter',
                include: [
                    {
                        model: Location,
                        as: 'location',
                        include: [
                            {
                                model: Facility,
                                as: 'facility'
                            }
                        ]
                    }
                ]
            },
            {
                model: ScheduledVaccine,
                as: 'scheduledVaccine',
                include: [
                    {
                        model: ReferenceData,
                        as: 'vaccine'
                    }
                ]
            }
        ]
    });
    _logging.log.debug('Translating VDS', {
        patientId,
        vaccinationCount: vaccinations.length
    });
    if (vaccinations.length === 0) throw new Error('Patient does not have any vaccinations');
    const pidDoc = passport ? {
        i: passport
    } : {};
    // Group by vaccine brand/label
    const vaccines = new Map();
    for (const dose of vaccinations){
        const { batch, date, location, scheduledVaccine: { doseLabel, vaccine: { name: label, id: vaccineId } }, encounter: { location: { facility: { name: encounterFacilityName } } } } = dose;
        const sublog = _logging.log.child({
            administeredVaccineId: dose.id,
            vaccineRefId: vaccineId
        });
        const facilityName = location?.facility?.name ?? encounterFacilityName;
        const certVax = await CertifiableVaccine.findOne({
            where: {
                vaccineId
            },
            include: [
                {
                    model: ReferenceData,
                    as: 'manufacturer'
                }
            ]
        });
        if (!certVax) {
            sublog.debug('Vaccine is not certifiable');
            continue;
        }
        const event = {
            dvc: (0, _datefnstz.formatInTimeZone)(date, timeZone, DATE_FORMAT_ISODATE),
            seq: SCHEDULE_TO_SEQUENCE[doseLabel],
            ctr: countryCode,
            lot: batch || 'Unknown',
            adm: facilityName
        };
        sublog.debug('Event for vaccine', {
            event
        });
        if (vaccines.has(label)) {
            sublog.debug('Adding to existing brand/label group', {
                label
            });
            const vax = vaccines.get(label);
            vax.vd.push(event);
            vaccines.set(label, vax);
        } else {
            sublog.debug('Adding to new brand/label group', {
                label
            });
            vaccines.set(label, {
                des: certVax.icd11DrugCode,
                nam: label,
                dis: certVax.icd11DiseaseCode,
                vd: [
                    event
                ]
            });
        }
    }
    _logging.log.debug('Translated VDS', {
        patientId,
        vaccinationCount: vaccines.size
    });
    if (vaccines.size === 0) throw new Error('No certifiable vaccines for patient');
    return {
        pid: {
            ...pid(firstName, lastName, dateOfBirth, sex),
            ...pidDoc
        },
        ve: [
            ...vaccines.values()
        ]
    };
};
const createVdsNcTestData = async (labTestId, { models })=>{
    const { Patient, PatientAdditionalData, LabTest, ReferenceData, LabRequest, Location, Encounter } = models;
    const { country } = await (0, _localisation.getLocalisation)();
    const countryCode = country['alpha-3'];
    const test = await LabTest.findOne({
        where: {
            id: labTestId
        },
        include: [
            {
                model: ReferenceData,
                as: 'labTestMethod'
            },
            {
                model: LabRequest,
                as: 'labRequest',
                include: [
                    {
                        model: Encounter,
                        as: 'encounter',
                        include: [
                            {
                                model: Patient,
                                as: 'patient',
                                include: [
                                    {
                                        model: PatientAdditionalData,
                                        as: 'additionalData'
                                    }
                                ]
                            },
                            {
                                model: Location,
                                as: 'location',
                                include: [
                                    'facility'
                                ]
                            }
                        ]
                    }
                ]
            }
        ]
    });
    const { labTestMethod: method, labRequest: request } = test;
    const { location: { facility }, patient: { firstName, lastName, dateOfBirth, sex, additionalData: [{ passport }] } } = request.encounter;
    const pidDoc = passport ? {
        dt: 'P',
        dn: passport
    } : {};
    return {
        pid: {
            ...pid(firstName, lastName, dateOfBirth, sex),
            ...pidDoc
        },
        sp: {
            spn: facility.name,
            ctr: countryCode,
            cd: {
                p: facility.contactNumber,
                e: facility.email,
                a: `${facility.streetAddress}, ${facility.cityTown}`
            }
        },
        dat: {
            sc: (0, _datefnstz.formatInTimeZone)(request.sampleTime, 'UTC', DATE_FORMAT_RFC3339),
            ri: (0, _datefnstz.formatInTimeZone)(new Date(test.completedDate), 'UTC', DATE_FORMAT_RFC3339)
        },
        tr: {
            tc: METHOD_CODE[method.code] ?? method.code,
            r: test.result
        }
    };
};
function pid(firstName, lastName, dob, sex) {
    const MAX_LEN = 39;
    const primary = (0, _transliteration.transliterate)(lastName);
    const secondary = (0, _transliteration.transliterate)(firstName);
    // Truncation from 9303-4, a bit simplified as we're not in MRZ
    let name;
    if (primary.length >= MAX_LEN - 3) {
        name = [
            primary.slice(0, MAX_LEN - 3),
            secondary
        ].join(' ').slice(0, MAX_LEN);
    } else {
        name = [
            primary,
            secondary
        ].join(' ').slice(0, MAX_LEN);
    }
    const data = {
        n: name,
        dob
    };
    if (sex && SEX_TO_CHAR[sex]) {
        data.sex = SEX_TO_CHAR[sex];
    }
    return data;
}

//# sourceMappingURL=Translation.js.map