"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "InvoiceItem", {
    enumerable: true,
    get: function() {
        return InvoiceItem;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("../Model");
const _buildEncounterLinkedSyncFilter = require("../../sync/buildEncounterLinkedSyncFilter");
const _model = require("../../types/model");
const _buildEncounterLinkedLookupFilter = require("../../sync/buildEncounterLinkedLookupFilter");
let InvoiceItem = class InvoiceItem extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            orderDate: (0, _model.dateType)('orderDate', {
                allowNull: false
            }),
            productId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            quantity: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: false
            },
            note: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            sourceRecordType: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            sourceRecordId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            productNameFinal: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            manualEntryPrice: {
                type: _sequelize.DataTypes.DECIMAL,
                allowNull: true
            },
            priceFinal: {
                type: _sequelize.DataTypes.DECIMAL,
                allowNull: true
            },
            productCodeFinal: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true
            },
            approved: {
                type: _sequelize.DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Invoice, {
            foreignKey: 'invoiceId',
            as: 'invoice'
        });
        this.hasOne(models.InvoiceItemDiscount, {
            foreignKey: 'invoiceItemId',
            as: 'discount'
        });
        this.belongsTo(models.User, {
            foreignKey: 'orderedByUserId',
            as: 'orderedByUser'
        });
        this.belongsTo(models.InvoiceProduct, {
            foreignKey: 'productId',
            as: 'product'
        });
        this.hasMany(models.InvoiceItemFinalisedInsurance, {
            foreignKey: 'invoiceItemId',
            as: 'finalisedInsurances'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'invoices',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return {
            select: await (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupSelect)(this),
            joins: (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupJoins)(this, [
                'invoices',
                'encounters'
            ])
        };
    }
    static getListReferenceAssociations(models, invoicePriceListId, tableAlias = 'InvoiceItem') {
        // Validate tableAlias against allow-list to prevent SQL injection
        const allowedAliases = [
            'InvoiceItem',
            'items'
        ];
        if (!allowedAliases.includes(tableAlias)) {
            throw new Error(`Invalid table alias: ${tableAlias}. Must be one of: ${allowedAliases.join(', ')}`);
        }
        const productInclude = [
            {
                model: models.ReferenceData,
                as: 'sourceRefDataRecord',
                attributes: [
                    'code',
                    'type'
                ]
            },
            {
                model: models.LabTestType,
                as: 'sourceLabTestTypeRecord',
                attributes: [
                    'code'
                ]
            },
            {
                model: models.LabTestPanel,
                as: 'sourceLabTestPanelRecord',
                attributes: [
                    'code'
                ]
            },
            {
                model: models.InvoiceInsurancePlanItem,
                as: 'invoiceInsurancePlanItems',
                required: false,
                where: {
                    [_sequelize.Op.and]: [
                        {
                            invoiceInsurancePlanId: {
                                [_sequelize.Op.in]: (0, _sequelize.literal)(`(
                  SELECT iip."invoice_insurance_plan_id"
                  FROM "invoices_invoice_insurance_plans" iip
                  WHERE iip."invoice_id" = "${tableAlias}"."invoice_id"
                    AND iip."deleted_at" IS NULL
                )`)
                            }
                        },
                        (0, _sequelize.literal)(`EXISTS (
              SELECT 1
              FROM "invoice_products" ip
              WHERE ip."id" = "${tableAlias}"."product_id"
                AND ip."insurable" = true
                AND ip."deleted_at" IS NULL
            )`)
                    ]
                }
            }
        ];
        if (invoicePriceListId) {
            productInclude.push({
                model: models.InvoicePriceListItem,
                where: {
                    invoicePriceListId
                },
                as: 'invoicePriceListItem',
                attributes: [
                    'price',
                    'invoicePriceListId'
                ],
                required: false
            });
        }
        return [
            {
                model: models.InvoiceProduct,
                as: 'product',
                include: productInclude
            },
            {
                model: models.User,
                as: 'orderedByUser',
                attributes: [
                    'displayName'
                ]
            },
            {
                model: models.InvoiceItemDiscount,
                as: 'discount'
            },
            {
                model: models.InvoiceItemFinalisedInsurance,
                as: 'finalisedInsurances'
            }
        ];
    }
};

//# sourceMappingURL=InvoiceItem.js.map