"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "Invoice", {
    enumerable: true,
    get: function() {
        return Invoice;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _Model = require("../Model");
const _buildEncounterLinkedSyncFilter = require("../../sync/buildEncounterLinkedSyncFilter");
const _buildEncounterLinkedLookupFilter = require("../../sync/buildEncounterLinkedLookupFilter");
const _model = require("../../types/model");
const _generateInvoiceDisplayId = require("@tamanu/utils/generateInvoiceDisplayId");
let Invoice = class Invoice extends _Model.Model {
    static initModel({ primaryKey, ...options }) {
        super.init({
            id: primaryKey,
            displayId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            date: (0, _model.dateTimeType)('date', {
                allowNull: false
            }),
            status: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false
            },
            patientPaymentStatus: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.INVOICE_PATIENT_PAYMENT_STATUSES.UNPAID
            },
            insurerPaymentStatus: {
                type: _sequelize.DataTypes.STRING,
                allowNull: false,
                defaultValue: _constants.INVOICE_INSURER_PAYMENT_STATUSES.UNPAID
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.Encounter, {
            foreignKey: 'encounterId',
            as: 'encounter'
        });
        this.hasOne(models.InvoiceDiscount, {
            foreignKey: 'invoiceId',
            as: 'discount'
        });
        this.hasMany(models.InvoicesInvoiceInsurancePlan, {
            foreignKey: 'invoiceId',
            as: 'invoiceInsurancePlans'
        });
        this.belongsToMany(models.InvoiceInsurancePlan, {
            through: models.InvoicesInvoiceInsurancePlan,
            foreignKey: 'invoiceId',
            otherKey: 'invoiceInsurancePlanId',
            as: 'insurancePlans'
        });
        this.hasMany(models.InvoiceItem, {
            foreignKey: 'invoiceId',
            as: 'items'
        });
        this.hasMany(models.InvoicePayment, {
            foreignKey: 'invoiceId',
            as: 'payments'
        });
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _buildEncounterLinkedSyncFilter.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return (0, _buildEncounterLinkedLookupFilter.buildEncounterLinkedLookupFilter)(this);
    }
    static getFullReferenceAssociations(invoicePriceListId) {
        const { models } = this.sequelize;
        return [
            'encounter',
            {
                model: models.InvoiceDiscount,
                as: 'discount',
                include: [
                    {
                        model: models.User,
                        as: 'appliedByUser',
                        attributes: [
                            'displayName'
                        ]
                    }
                ]
            },
            {
                model: models.InvoiceItem,
                as: 'items',
                include: models.InvoiceItem.getListReferenceAssociations(models, invoicePriceListId, 'items')
            },
            {
                model: models.InvoicePayment,
                as: 'payments',
                include: models.InvoicePayment.getListReferenceAssociations(models)
            },
            {
                model: models.InvoiceInsurancePlan,
                as: 'insurancePlans'
            }
        ];
    }
    static async getInProgressInvoiceForEncounter(encounterId) {
        const invoices = await this.findAll({
            where: {
                encounterId,
                status: _constants.INVOICE_STATUSES.IN_PROGRESS
            }
        });
        if (invoices.length === 0) {
            return null; // No in progress invoice for encounter
        }
        if (invoices.length > 1) {
            throw new Error(`Multiple in progress invoices found for encounter: ${encounterId}`);
        }
        return invoices[0];
    }
    static async addItemToInvoice(newItem, encounterId, invoiceProduct, orderedByUserId = _constants.SYSTEM_USER_UUID, note) {
        const invoice = await this.getInProgressInvoiceForEncounter(encounterId);
        if (!invoice) {
            return;
        }
        const invoicePriceListId = await this.sequelize.models.InvoicePriceList.getIdForPatientEncounter(encounterId);
        if (invoicePriceListId) {
            // Confirm invoice product is not configured to be hidden for this price list
            const hiddenInvoicePriceListItem = await this.sequelize.models.InvoicePriceListItem.findOne({
                where: {
                    invoicePriceListId,
                    invoiceProductId: invoiceProduct.id,
                    isHidden: true
                }
            });
            if (hiddenInvoicePriceListItem) {
                return;
            }
        }
        await this.sequelize.models.InvoiceItem.upsert({
            invoiceId: invoice.id,
            sourceRecordType: newItem.getModelName(),
            sourceRecordId: newItem.id,
            productId: invoiceProduct.id,
            orderedByUserId,
            orderDate: new Date(),
            quantity: 1,
            note,
            deletedAt: null
        }, {
            conflictFields: [
                'invoice_id',
                'source_record_type',
                'source_record_id'
            ]
        });
    }
    static async removeItemFromInvoice(removedItemSource, encounterId) {
        const invoice = await this.getInProgressInvoiceForEncounter(encounterId);
        if (!invoice) {
            return;
        }
        await this.sequelize.models.InvoiceItem.destroy({
            where: {
                invoiceId: invoice.id,
                sourceRecordType: removedItemSource.getModelName(),
                sourceRecordId: removedItemSource.id
            }
        });
    }
    static async automaticallyCreateForEncounter(encounterId, encounterType, date, settings, options) {
        const isInvoicingEnabled = await settings?.get('features.invoicing.enabled');
        const isValidEncounterType = !_constants.AUTOMATIC_INVOICE_CREATION_EXCLUDED_ENCOUNTER_TYPES.includes(encounterType);
        if (!isInvoicingEnabled || !isValidEncounterType) {
            return null;
        }
        return await this.create({
            displayId: (0, _generateInvoiceDisplayId.generateInvoiceDisplayId)(),
            status: _constants.INVOICE_STATUSES.IN_PROGRESS,
            date,
            encounterId
        }, options);
    }
};

//# sourceMappingURL=Invoice.js.map