import { DeviceScope } from '@tamanu/constants';
import { BaseFetchOptions } from './fetch';
import { RetryBackoffOptions } from './fetchWithRetryBackoff';
import { InterceptorManager, type RequestInterceptorFulfilled, type RequestInterceptorRejected, type ResponseInterceptorFulfilled, type ResponseInterceptorRejected } from './InterceptorManager';
interface Logger {
    debug: (message: string, data?: any) => void;
    warn: (message: string, data?: any) => void;
    error: (message: string, data?: any) => void;
    info: (message: string, data?: any) => void;
    log: (message: string, data?: any) => void;
}
export type LoggerType = Logger | Console;
interface User {
    id: string;
    email: string;
    displayName: string;
}
interface LoginData {
    token: string;
    refreshToken: string;
    permissions?: string[];
}
interface LoginResponse extends LoginData {
    user: User;
    ability: {
        can: (action: string, subject: string, field?: string) => boolean;
    };
    server: ServerInfo;
}
interface ServerInfo {
    type: string;
    centralHost?: string;
}
interface TamanuApiConfig {
    endpoint: string;
    agentName: string;
    agentVersion: string;
    deviceId: string;
    defaultRequestConfig?: RequestInit;
    logger?: LoggerType;
}
interface FetchOptions extends BaseFetchOptions {
    useAuthToken?: string | boolean;
    returnResponse?: boolean;
    throwResponse?: boolean;
    waitForAuth?: boolean;
    backoff?: boolean | RetryBackoffOptions;
    scopes?: DeviceScope[];
    body?: any;
}
interface PasswordChangeArgs {
    currentPassword: string;
    newPassword: string;
}
interface StreamEndpointConfig {
    endpoint: string;
    query?: Record<string, any>;
    options?: Record<string, any>;
}
interface StreamOptions {
    decodeMessage?: boolean;
    streamRetryAttempts?: number;
    streamRetryInterval?: number;
}
interface StreamMessage {
    kind: number;
    message: any;
}
export declare class TamanuApi {
    #private;
    lastRefreshed: number | null;
    user: User | null;
    logger: LoggerType;
    fetchImplementation: typeof fetch;
    agentName: string;
    agentVersion: string;
    deviceId: string;
    interceptors: {
        request: InterceptorManager<RequestInterceptorFulfilled, RequestInterceptorRejected>;
        response: InterceptorManager<ResponseInterceptorFulfilled, ResponseInterceptorRejected>;
    };
    constructor({ endpoint, agentName, agentVersion, deviceId, defaultRequestConfig, logger, }: TamanuApiConfig);
    get host(): string;
    setAuthFailureHandler(handler: (message: string) => void): void;
    setVersionIncompatibleHandler(handler: (message: string) => void): void;
    login(email: string, password: string, config?: FetchOptions): Promise<LoginResponse>;
    fetchUserData(permissions: string[], config?: FetchOptions): Promise<{
        user: User;
        ability: any;
    }>;
    requestPasswordReset(email: string): Promise<any>;
    changePassword(args: PasswordChangeArgs): Promise<any>;
    refreshToken(config?: FetchOptions): Promise<void>;
    setToken(token: string, refreshToken?: string | null): void;
    hasToken(): boolean;
    fetch(endpoint: string, query?: Record<string, any>, options?: FetchOptions): Promise<any>;
    get<T = any>(endpoint: string, query?: Record<string, any>, config?: FetchOptions): Promise<T>;
    download(endpoint: string, query?: Record<string, any>): Promise<Blob>;
    postWithFileUpload(endpoint: string, file: Blob, body: any, options?: FetchOptions): Promise<any>;
    post<T = any>(endpoint: string, body?: any, config?: FetchOptions): Promise<T>;
    put<T = any>(endpoint: string, body?: any, config?: FetchOptions): Promise<T>;
    delete<T = any>(endpoint: string, query?: Record<string, any>, config?: FetchOptions): Promise<T>;
    pollUntilOk<T = any>(endpoint: string, query?: Record<string, any>, config?: FetchOptions): Promise<T>;
    /** Connect to a streaming endpoint and async yield messages.
     *
     * ```js
     * for await (const { kind, message } of centralServer.stream(() => ({
     *   endpoint: `some/kind/of/stream`,
     * }))) {
     *   switch (kind) {
     *     case SYNC_STREAM_MESSAGE_KIND.SOMETHING:
     *       // do something
     *       break;
     *     case: SYNC_STREAM_MESSAGE_KIND.END:
     *       // finalise
     *       break;
     *     default:
     *       console.warn(`Unknown message kind: ${kind}`);
     *   }
     * }
     * ```
     *
     * The streaming endpoint needs to talk the Tamanu Streaming Protocol: a lightweight framing
     * protocol which includes a 2-byte Message Kind (unsigned int) and an optional JSON payload.
     * The stream MUST end with an `END` Message Kind (which may have a payload): if the stream
     * does not receive an `END` message, it is assumed to be incomplete and is automatically
     * restarted; this protects against unexpected stream disconnections.
     *
     * There are two possible layers of retry logic: on connection, using the endpointFn `options`
     * map, you can set `backoff` to retry the fetch on initial failure. This applies on top of the
     * stream retries, controlled by `streamRetryAttempts` (default 10) and `streamRetryInterval`
     * (milliseconds, default 10 seconds), which will restart the entire stream if it fails early.
     * Set `streamRetryAttempts` to 1 to disable the retries.
     *
     * Because the entire stream is restarted during a stream retry, the endpoint is not a fixed URL
     * but instead a function which is expected to return an object: `{ endpoint, query, options }`.
     * The `endpoint` key is required, the others default to `{}` if not present. These are passed to
     * and interpreted the same as for `.fetch()` above.
     *
     * For example, you can track some progress information from the messages you receive, and then
     * provide a "start from this point" query parameter to the next retry call. This avoids either
     * receiving the full stream contents again or keeping track of stream session state server-side.
     *
     * Message payloads are expected to be JSON, and by default are parsed directly within this
     * function. If you expect non-JSON payloads, or if you want to obtain the raw payload for some
     * other reason, pass `decodeMessage: false`. This will be slightly faster as the framing allows
     * us to seek forward through the received data rather than read every byte.
     *
     * @param endpointFn Function that returns endpoint configuration
     * @param streamOptions Stream configuration options
     * @returns AsyncGenerator yielding stream messages
     */
    stream(endpointFn: () => StreamEndpointConfig, { decodeMessage, streamRetryAttempts, streamRetryInterval, }?: StreamOptions): AsyncGenerator<StreamMessage, void, unknown>;
}
export {};
