"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "UserLoginAttempt", {
    enumerable: true,
    get: function() {
        return UserLoginAttempt;
    }
});
const _constants = require("@tamanu/constants");
const _Model = require("./Model");
const _sequelize = require("sequelize");
const _logging = require("@tamanu/shared/services/logging");
let UserLoginAttempt = class UserLoginAttempt extends _Model.Model {
    static initModel(options) {
        super.init({
            id: {
                type: _sequelize.DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: _sequelize.Sequelize.fn('gen_random_uuid')
            },
            outcome: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: false
            },
            deviceId: {
                type: _sequelize.DataTypes.TEXT,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.PUSH_TO_CENTRAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'userId',
            as: 'user'
        });
    }
    // If the deviceId is not found in the devices table, we will use null
    // for the deviceId to avoid credential stuffing from unregistered devices.
    static async getDeviceIdToUse(deviceId) {
        const device = await this.sequelize.models.Device.findByPk(deviceId);
        return device ? deviceId : null;
    }
    static async checkIsUserLockedOut({ settings, userId, deviceId = '' }) {
        const { lockoutDuration } = await settings.get(_constants.SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const deviceIdToUse = await this.getDeviceIdToUse(deviceId);
        const lockedAttempt = await this.findOne({
            where: {
                userId,
                deviceId: deviceIdToUse,
                outcome: _constants.LOGIN_ATTEMPT_OUTCOMES.LOCKED,
                createdAt: {
                    [_sequelize.Op.gte]: _sequelize.Sequelize.literal("CURRENT_TIMESTAMP - $lockoutDuration * interval '1 minute'")
                }
            },
            order: [
                [
                    'createdAt',
                    'DESC'
                ]
            ],
            bind: {
                lockoutDuration
            }
        });
        const createdAt = lockedAttempt?.createdAt?.getTime() ?? 0;
        const remainingLockout = lockoutDuration * 60 - Math.floor((Date.now() - createdAt) / 1000);
        return {
            isUserLockedOut: !!lockedAttempt,
            remainingLockout: Math.max(0, remainingLockout)
        };
    }
    static async createFailedLoginAttempt({ settings, userId, deviceId = '' }) {
        const { lockoutThreshold, observationWindow, lockoutDuration } = await settings.get(_constants.SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const deviceIdToUse = await this.getDeviceIdToUse(deviceId);
        const failedAttempts = await this.count({
            where: {
                userId,
                deviceId: deviceIdToUse,
                outcome: _constants.LOGIN_ATTEMPT_OUTCOMES.FAILED,
                createdAt: {
                    [_sequelize.Op.gte]: _sequelize.Sequelize.literal("CURRENT_TIMESTAMP - $observationWindow * interval '1 minute'")
                }
            },
            // @ts-ignore - sequelize doesn't know bind works in count
            bind: {
                observationWindow
            }
        });
        // We need to add 1 to the failed attempts because the current attempt is not included in the count
        const outcome = failedAttempts + 1 >= lockoutThreshold ? _constants.LOGIN_ATTEMPT_OUTCOMES.LOCKED : _constants.LOGIN_ATTEMPT_OUTCOMES.FAILED;
        let loginAttempt = null;
        try {
            loginAttempt = await this.create({
                userId,
                deviceId: deviceIdToUse,
                outcome
            });
        } catch (error) {
            _logging.log.error('Error creating failed login attempt', error);
        }
        return {
            loginAttempt,
            lockoutDuration: lockoutDuration * 60,
            remainingAttempts: Math.max(0, lockoutThreshold - failedAttempts - 1)
        };
    }
};

//# sourceMappingURL=UserLoginAttempt.js.map