"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "encounter", {
    enumerable: true,
    get: function() {
        return encounter;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _ability = require("@casl/ability");
const _errors = require("@tamanu/errors");
const _dateTime = require("@tamanu/utils/dateTime");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _countryDateTime = require("@tamanu/shared/utils/countryDateTime");
const _constants = require("@tamanu/constants");
const _crudHelpers = require("@tamanu/shared/utils/crudHelpers");
const _datefns = require("date-fns");
const _zod = require("zod");
const _charts = require("../../routeHandlers/charts");
const _lodash = require("lodash");
const _createEncounterschema = require("@tamanu/shared/schemas/facility/requests/createEncounter.schema");
const _uploadAttachment = require("../../utils/uploadAttachment");
const _routeHandlers = require("../../routeHandlers");
const _createPatientLetter = require("../../routeHandlers/createPatientLetter");
const _labs = require("../../routeHandlers/labs");
const _deleteModel = require("../../routeHandlers/deleteModel");
const _getPermittedSurveyIds = require("../../utils/getPermittedSurveyIds");
const _validate = require("../../utils/validate");
const _invoiceForResponse = require("./invoice/invoiceForResponse");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const encounter = (0, _crudHelpers.softDeletionCheckingRouter)('Encounter');
encounter.get('/:id', (0, _crudHelpers.simpleGet)('Encounter', {
    auditAccess: true
}));
encounter.post('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body: { facilityId, ...data }, user } = req;
    req.checkPermission('create', 'Encounter');
    const validatedBody = (0, _validate.validate)(_createEncounterschema.createEncounterSchema, data);
    if (!validatedBody.endDate) {
        const existingOpenEncounterCount = await models.Encounter.count({
            where: {
                patientId: validatedBody.patientId,
                endDate: null,
                deletedAt: null
            }
        });
        if (existingOpenEncounterCount > 0) {
            throw new _errors.InvalidOperationError('This patient already has an active encounter. The active encounter must be discharged before a new active encounter can be created.');
        }
    }
    const encounterObject = await models.Encounter.create({
        ...validatedBody,
        actorId: user.id
    });
    await models.Invoice.automaticallyCreateForEncounter(encounterObject.id, encounterObject.encounterType, encounterObject.startDate, req.settings[facilityId]);
    if (data.dietIds) {
        const dietIds = JSON.parse(data.dietIds);
        await encounterObject.addDiets(dietIds);
    }
    res.send(encounterObject);
}));
encounter.put('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, user, params } = req;
    const { referralId, id } = params;
    req.checkPermission('read', 'Encounter');
    const encounterObject = await models.Encounter.findByPk(id);
    if (!encounterObject) throw new _errors.NotFoundError();
    req.checkPermission('write', encounterObject);
    await db.transaction(async ()=>{
        let systemNote;
        if (req.body.discharge) {
            req.checkPermission('write', 'Discharge');
            if (!req.body.discharge.dischargerId) {
                // Only automatic discharges can have a null discharger ID
                throw new _errors.InvalidParameterError('A discharge must have a discharger.');
            }
            const discharger = await models.User.findByPk(req.body.discharge.dischargerId);
            if (!discharger) {
                throw new _errors.InvalidParameterError(`Discharger with id ${req.body.discharge.dischargerId} not found.`);
            }
            systemNote = `Patient discharged by ${discharger.displayName}.`;
            const prescriptions = req.body.medications || {};
            for (const [prescriptionId, prescriptionValues] of Object.entries(prescriptions)){
                const { quantity, repeats } = prescriptionValues;
                const prescription = await models.Prescription.findByPk(prescriptionId, {
                    include: [
                        {
                            model: models.EncounterPrescription,
                            as: 'encounterPrescription',
                            attributes: [
                                'encounterId'
                            ],
                            required: false
                        },
                        {
                            model: models.PatientOngoingPrescription,
                            as: 'patientOngoingPrescription',
                            attributes: [
                                'patientId'
                            ],
                            required: false
                        }
                    ]
                });
                if (!prescription || prescription.discontinued) continue;
                await prescription.update({
                    quantity,
                    repeats
                });
                await models.EncounterPrescription.update({
                    isSelectedForDischarge: true
                }, {
                    where: {
                        encounterId: id,
                        prescriptionId: prescription.id
                    }
                });
                // If the medication is ongoing and not already in the patient's ongoing medications, we need to add it to the patient's ongoing medications
                if (prescription.isOngoing && prescription.encounterPrescription?.encounterId === id) {
                    const existingPatientOngoingPrescription = await models.PatientOngoingPrescription.findPatientOngoingPrescriptionWithSameDetails(encounterObject.patientId, prescription);
                    if (existingPatientOngoingPrescription) continue;
                    await models.PatientOngoingPrescription.create({
                        patientId: encounterObject.patientId,
                        prescriptionId: prescription.id
                    });
                }
            }
        }
        if (referralId) {
            const referral = await models.Referral.findByPk(referralId, {
                paranoid: false
            });
            if (referral && referral.deletedAt) throw new _errors.InvalidOperationError('Cannot update a deleted referral.');
            await referral.update({
                encounterId: id
            });
        }
        if (req.body.locationId != null) {
            const location = await models.Location.findByPk(req.body.locationId);
            if (!location) {
                throw new _errors.InvalidOperationError('Invalid location specified');
            }
        }
        await encounterObject.update({
            ...req.body,
            systemNote
        }, user);
        if (req.body.dietIds) {
            const dietIds = JSON.parse(req.body.dietIds);
            await encounterObject.setDiets(dietIds);
        }
    });
    res.send(encounterObject);
}));
encounter.post('/:id/notes', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, body, params } = req;
    const { id } = params;
    req.checkPermission('write', 'Encounter');
    const owner = await models.Encounter.findByPk(id);
    if (!owner) {
        throw new _errors.NotFoundError();
    }
    req.checkPermission('write', owner);
    const note = await owner.createNote(body);
    res.send(note);
}));
encounter.post('/:id/documentMetadata', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    // TODO: figure out permissions with Attachment and DocumentMetadata
    req.checkPermission('write', 'DocumentMetadata');
    // Make sure the specified encounter exists
    const specifiedEncounter = await models.Encounter.findByPk(params.id);
    if (!specifiedEncounter) {
        throw new _errors.NotFoundError();
    }
    // Create file on the central server
    const { attachmentId, type, metadata } = await (0, _uploadAttachment.uploadAttachment)(req, _constants.DOCUMENT_SIZE_LIMIT);
    const documentMetadataObject = await models.DocumentMetadata.create({
        ...metadata,
        attachmentId,
        type,
        encounterId: params.id,
        documentUploadedAt: (0, _dateTime.getCurrentDateTimeString)(),
        source: _constants.DOCUMENT_SOURCES.UPLOADED
    });
    res.send(documentMetadataObject);
}));
encounter.post('/:id/createPatientLetter', (0, _createPatientLetter.createPatientLetter)('Encounter', 'encounterId'));
encounter.post('/:id/pharmacyOrder', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, params, body } = req;
    const { id } = params;
    req.checkPermission('write', 'Encounter');
    req.checkPermission('read', 'Medication');
    req.checkPermission('create', 'MedicationRequest');
    const encounterObject = await models.Encounter.findByPk(id);
    if (!encounterObject) throw new _errors.NotFoundError();
    const { orderingClinicianId, comments, isDischargePrescription, pharmacyOrderPrescriptions, facilityId } = body;
    const prescriptionRecords = await models.Prescription.findAll({
        where: {
            id: pharmacyOrderPrescriptions.map((p)=>p.prescriptionId)
        },
        attributes: [
            'id',
            'medicationId',
            'quantity'
        ]
    });
    const hasSensitive = await models.ReferenceDrug.hasSensitiveMedication(prescriptionRecords.map((p)=>p.medicationId));
    if (hasSensitive) {
        req.checkPermission('read', 'SensitiveMedication');
    }
    const result = await db.transaction(async ()=>{
        const pharmacyOrder = await models.PharmacyOrder.create({
            orderingClinicianId,
            encounterId: id,
            comments,
            isDischargePrescription,
            date: (0, _dateTime.getCurrentDateTimeString)(),
            facilityId
        });
        await models.PharmacyOrderPrescription.bulkCreate(pharmacyOrderPrescriptions.map((prescription)=>({
                pharmacyOrderId: pharmacyOrder.id,
                prescriptionId: prescription.prescriptionId,
                quantity: prescription.quantity,
                repeats: prescription.repeats
            })));
        return pharmacyOrder;
    });
    res.send(result);
}));
encounter.delete('/:id/documentMetadata/:documentMetadataId', _deleteModel.deleteDocumentMetadata);
encounter.delete('/:id', _deleteModel.deleteEncounter);
const encounterRelations = (0, _crudHelpers.permissionCheckingRouter)('read', 'Encounter');
encounterRelations.get('/:id/discharge', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models: { Discharge }, params } = req;
    req.checkPermission('read', 'Discharge');
    const discharge = await Discharge.findOne({
        where: {
            encounterId: params.id
        },
        include: Discharge.getFullReferenceAssociations()
    });
    if (!discharge) throw new _errors.NotFoundError();
    await req.audit?.access?.({
        recordId: discharge.id,
        frontEndContext: params,
        model: Discharge
    });
    const plain = discharge.get({
        plain: true
    });
    plain.address = await discharge.address();
    res.send(plain);
}));
encounterRelations.get('/:id/legacyVitals', (0, _crudHelpers.simpleGetList)('Vitals', 'encounterId'));
encounterRelations.get('/:id/diagnoses', (0, _crudHelpers.simpleGetList)('EncounterDiagnosis', 'encounterId'));
encounterRelations.get('/:id/medications', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query, db } = req;
    const { Prescription } = models;
    const { order = 'ASC', orderBy = 'medication.name', rowsPerPage, page, marDate } = query;
    req.checkPermission('list', 'Medication');
    const associations = Prescription.getListReferenceAssociations() || [];
    const medicationFilter = {};
    const canListSensitiveMedication = req.ability.can('list', 'SensitiveMedication');
    if (!canListSensitiveMedication) {
        medicationFilter['$medication.referenceDrug.is_sensitive$'] = false;
    }
    const baseQueryOptions = {
        where: medicationFilter,
        order: [
            [
                (0, _sequelize.literal)('CASE WHEN "discontinued" IS NULL OR "discontinued" = false THEN 1 ELSE 0 END'),
                'DESC'
            ],
            ...orderBy ? [
                [
                    ...orderBy.split('.'),
                    order.toUpperCase()
                ]
            ] : [],
            [
                'date',
                'ASC'
            ]
        ],
        include: [
            ...associations,
            {
                model: models.EncounterPrescription,
                as: 'encounterPrescription',
                include: {
                    model: models.EncounterPausePrescription,
                    as: 'pausePrescriptions',
                    attributes: [
                        'pauseDuration',
                        'pauseTimeUnit',
                        'pauseEndDate'
                    ],
                    where: {
                        pauseEndDate: {
                            [_sequelize.Op.gt]: (0, _dateTime.getCurrentDateTimeString)()
                        }
                    },
                    required: false
                },
                attributes: [
                    'id',
                    'encounterId',
                    'isSelectedForDischarge'
                ],
                where: {
                    encounterId: params.id
                }
            },
            {
                model: models.ReferenceData,
                as: 'medication',
                include: {
                    model: models.ReferenceDrug,
                    as: 'referenceDrug',
                    attributes: [
                        'referenceDataId',
                        'isSensitive'
                    ]
                }
            }
        ]
    };
    // Add medicationAdministrationRecords with condition for same day
    if (marDate) {
        req.checkPermission('list', 'MedicationAdministration');
        const startOfMarDate = `${marDate} 00:00:00`;
        const endOfMarDate = `${marDate} 23:59:59`;
        baseQueryOptions.include.push({
            model: models.MedicationAdministrationRecord,
            as: 'medicationAdministrationRecords',
            where: {
                dueAt: {
                    [_sequelize.Op.gte]: startOfMarDate,
                    [_sequelize.Op.lte]: endOfMarDate
                }
            },
            include: [
                {
                    association: 'reasonNotGiven',
                    attributes: [
                        'id',
                        'name',
                        'type'
                    ]
                },
                {
                    association: 'recordedByUser',
                    attributes: [
                        'id',
                        'displayName'
                    ]
                }
            ],
            required: false
        });
        baseQueryOptions.where = {
            ...baseQueryOptions.where,
            startDate: {
                [_sequelize.Op.lte]: endOfMarDate
            },
            [_sequelize.Op.and]: [
                {
                    [_sequelize.Op.or]: [
                        {
                            discontinuedDate: {
                                [_sequelize.Op.is]: null
                            }
                        },
                        {
                            discontinuedDate: {
                                [_sequelize.Op.gt]: startOfMarDate
                            }
                        }
                    ]
                },
                {
                    [_sequelize.Op.or]: [
                        {
                            endDate: null
                        },
                        {
                            endDate: {
                                [_sequelize.Op.gt]: startOfMarDate
                            }
                        }
                    ]
                }
            ]
        };
    }
    const count = await Prescription.count({
        ...baseQueryOptions,
        distinct: true
    });
    const prescriptions = await Prescription.findAll({
        ...baseQueryOptions,
        limit: rowsPerPage,
        offset: page && rowsPerPage ? page * rowsPerPage : undefined
    });
    let responseData = prescriptions.map((p)=>p.forResponse());
    if (responseData.length > 0) {
        const prescriptionIds = responseData.map((p)=>p.id);
        const [lastOrderedRows] = await db.query(`
        SELECT pop.prescription_id, max(po.date) AS last_ordered_at
        FROM pharmacy_order_prescriptions pop
        INNER JOIN pharmacy_orders po ON po.id = pop.pharmacy_order_id
        WHERE pop.prescription_id IN (:prescriptionIds)
          AND pop.deleted_at IS NULL
          AND po.deleted_at IS NULL
        GROUP BY pop.prescription_id
      `, {
            replacements: {
                prescriptionIds
            }
        });
        const lastOrderedAts = (0, _lodash.keyBy)(lastOrderedRows, 'prescription_id');
        responseData = responseData.map((p)=>({
                ...p,
                lastOrderedAt: lastOrderedAts[p.id]?.last_ordered_at
            }));
    }
    res.send({
        count,
        data: responseData
    });
}));
encounterRelations.get('/:id/procedures', (0, _crudHelpers.simpleGetList)('Procedure', 'encounterId'));
encounterRelations.get('/:id/labRequests', (0, _labs.getLabRequestList)('encounterId', {
    additionalFilters: {
        status: {
            [_sequelize.Op.notIn]: [
                _constants.LAB_REQUEST_STATUSES.DELETED,
                _constants.LAB_REQUEST_STATUSES.ENTERED_IN_ERROR
            ]
        }
    }
}));
encounterRelations.get('/:id/referral', (0, _crudHelpers.simpleGetList)('Referral', 'encounterId'));
encounterRelations.get('/:id/triages', (0, _crudHelpers.simpleGetList)('Triage', 'encounterId'));
encounterRelations.get('/:id/documentMetadata', (0, _crudHelpers.paginatedGetList)('DocumentMetadata', 'encounterId'));
encounterRelations.get('/:id/imagingRequests', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params, query } = req;
    const { ImagingRequest } = models;
    const { id: encounterId } = params;
    const { order = 'ASC', orderBy = 'createdAt', rowsPerPage, page, includeNotes: includeNotesStr = 'true', status } = query;
    const includeNote = includeNotesStr === 'true';
    req.checkPermission('list', 'ImagingRequest');
    const associations = ImagingRequest.getListReferenceAssociations() || [];
    const baseQueryOptions = {
        where: {
            encounterId,
            status: status || {
                [_sequelize.Op.notIn]: [
                    _constants.IMAGING_REQUEST_STATUS_TYPES.DELETED,
                    _constants.IMAGING_REQUEST_STATUS_TYPES.ENTERED_IN_ERROR
                ]
            }
        },
        order: orderBy ? [
            [
                ...orderBy.split('.'),
                order.toUpperCase()
            ]
        ] : undefined,
        include: associations
    };
    const count = await ImagingRequest.count({
        ...baseQueryOptions
    });
    const objects = await ImagingRequest.findAll({
        ...baseQueryOptions,
        limit: rowsPerPage,
        offset: page && rowsPerPage ? page * rowsPerPage : undefined
    });
    const data = await Promise.all(objects.map(async (ir)=>{
        return {
            ...ir.forResponse(),
            ...includeNote ? await ir.extractNotes() : undefined,
            areas: ir.areas.map((a)=>a.forResponse()),
            results: ir.results.map((result)=>result.forResponse())
        };
    }));
    res.send({
        count,
        data
    });
}));
encounterRelations.get('/:id/notes', (0, _routeHandlers.noteListHandler)(_constants.NOTE_RECORD_TYPES.ENCOUNTER));
encounterRelations.get('/:id/notes/noteTypes', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const encounterId = params.id;
    const noteTypeCounts = await models.Note.count({
        group: [
            'noteTypeId'
        ],
        where: {
            recordId: encounterId,
            recordType: 'Encounter'
        }
    });
    const noteTypeToCount = {};
    noteTypeCounts.forEach((n)=>{
        noteTypeToCount[n.noteTypeId] = n.count;
    });
    res.send({
        data: noteTypeToCount
    });
}));
encounterRelations.get('/:id/notes/:noteId/changelogs', (0, _routeHandlers.noteChangelogsHandler)(_constants.NOTE_RECORD_TYPES.ENCOUNTER));
encounterRelations.get('/:id/invoice', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Invoice, InvoicePriceList } = models;
    req.checkPermission('read', 'Invoice');
    const encounterId = params.id;
    const invoicePriceListId = await InvoicePriceList.getIdForPatientEncounter(encounterId);
    const invoiceRecord = await Invoice.findOne({
        where: {
            encounterId
        },
        include: Invoice.getFullReferenceAssociations(invoicePriceListId),
        order: [
            [
                {
                    model: models.InvoiceItem,
                    as: 'items'
                },
                'orderDate',
                'ASC'
            ],
            [
                {
                    model: models.InvoiceItem,
                    as: 'items'
                },
                'createdAt',
                'ASC'
            ],
            [
                {
                    model: models.InvoicePayment,
                    as: 'payments'
                },
                'date',
                'ASC'
            ]
        ]
    });
    if (!invoiceRecord) {
        // Return null rather than a 404 as it is a valid scenario for there not to be an invoice
        return res.send(null);
    }
    await req.audit.access({
        recordId: invoiceRecord.id,
        frontEndContext: params,
        model: Invoice
    });
    const responseRecord = (0, _invoiceForResponse.invoiceForResponse)(invoiceRecord);
    const priceList = await InvoicePriceList.findByPk(invoicePriceListId, {
        raw: true
    });
    res.send({
        ...responseRecord,
        priceList
    });
}));
const PROGRAM_RESPONSE_SORT_KEYS = {
    endTime: 'end_time',
    submittedBy: 'submitted_by',
    programName: 'program_name',
    surveyName: 'survey_name',
    resultText: 'result_text'
};
encounterRelations.get('/:id/programResponses', (0, _expressasynchandler.default)(async (req, res)=>{
    const { db, models, params, query } = req;
    req.checkPermission('list', 'SurveyResponse');
    const encounterId = params.id;
    const surveyType = 'programs';
    const { order = 'asc', orderBy = 'endTime' } = query;
    const sortKey = PROGRAM_RESPONSE_SORT_KEYS[orderBy] || PROGRAM_RESPONSE_SORT_KEYS.endTime;
    const sortDirection = order.toLowerCase() === 'asc' ? 'ASC' : 'DESC';
    const permittedSurveyIds = await (0, _getPermittedSurveyIds.getPermittedSurveyIds)(req, models);
    if (!permittedSurveyIds.length) {
        res.send({
            data: [],
            count: 0
        });
    }
    const { count, data } = await (0, _crudHelpers.runPaginatedQuery)(db, models.SurveyResponse, `
        SELECT COUNT(1) as count
        FROM
          survey_responses
          LEFT JOIN encounters
            ON (survey_responses.encounter_id = encounters.id)
          LEFT JOIN surveys
            ON (survey_responses.survey_id = surveys.id)
        WHERE
          survey_responses.encounter_id = :encounterId
        AND
          surveys.survey_type = :surveyType
        AND
          surveys.id IN (:surveyIds)
        AND
          encounters.deleted_at IS NULL
        AND
          survey_responses.deleted_at IS NULL
      `, `
        SELECT
          survey_responses.*,
          surveys.name as survey_name,
          programs.name as program_name,
          COALESCE(survey_user.display_name, encounter_user.display_name) as submitted_by
        FROM
          survey_responses
          LEFT JOIN surveys
            ON (survey_responses.survey_id = surveys.id)
          LEFT JOIN programs
            ON (programs.id = surveys.program_id)
          LEFT JOIN encounters
            ON (encounters.id = survey_responses.encounter_id)
          LEFT JOIN users encounter_user
            ON (encounter_user.id = encounters.examiner_id)
          LEFT JOIN users survey_user
            ON (survey_user.id = survey_responses.user_id)
        WHERE
          survey_responses.encounter_id = :encounterId
        AND
          surveys.survey_type = :surveyType
        AND
          surveys.id IN (:surveyIds)
        AND
          survey_responses.deleted_at IS NULL
        AND
          encounters.deleted_at is null
        AND survey_responses.id NOT IN (SELECT survey_response_id FROM procedure_survey_responses)
        ORDER BY ${sortKey} ${sortDirection}
      `, {
        encounterId,
        surveyType,
        surveyIds: permittedSurveyIds
    }, query);
    res.send({
        count: parseInt(count, 10),
        data
    });
}));
encounterRelations.delete('/:id/programResponses/:surveyResponseId', _deleteModel.deleteSurveyResponse);
encounterRelations.get('/:id/vitals', (0, _charts.fetchAnswersWithHistory)({
    permissionAction: 'list',
    permissionNoun: 'Vitals'
}));
encounterRelations.get('/:id/graphData/vitals/:dataElementId', (0, _charts.fetchGraphData)({
    permissionAction: 'list',
    permissionNoun: 'Vitals',
    dateDataElementId: _constants.VITALS_DATA_ELEMENT_IDS.dateRecorded
}));
encounterRelations.get('/:id/initialChart$', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { id: encounterId } = params;
    const chartSurvey = await models.SurveyResponse.findAll({
        attributes: [
            'survey.*'
        ],
        where: {
            encounterId
        },
        include: [
            {
                attributes: [
                    'id',
                    'name'
                ],
                required: true,
                model: models.Survey,
                as: 'survey',
                where: {
                    surveyType: [
                        _constants.SURVEY_TYPES.SIMPLE_CHART,
                        _constants.SURVEY_TYPES.COMPLEX_CHART
                    ]
                }
            }
        ],
        order: [
            [
                'survey',
                'name',
                'ASC'
            ]
        ],
        group: [
            [
                'survey.id'
            ]
        ]
    });
    req.flagPermissionChecked();
    const allowedSurvey = chartSurvey.find((response)=>req.ability.can('list', (0, _ability.subject)('Charting', {
            id: response.survey.id
        })));
    res.send({
        data: allowedSurvey
    });
}));
encounterRelations.get('/:id/graphData/charts/:dataElementId', (0, _charts.fetchGraphData)({
    permissionAction: 'read',
    permissionNoun: 'Charting',
    dateDataElementId: _constants.CHARTING_DATA_ELEMENT_IDS.dateRecorded
}));
encounterRelations.get('/:id/charts/:surveyId', (0, _charts.fetchAnswersWithHistory)({
    permissionAction: 'read',
    permissionNoun: 'Charting'
}));
const encounterTasksQuerySchema = _zod.z.object({
    order: _zod.z.preprocess((value)=>typeof value === 'string' ? value.toUpperCase() : value, _zod.z.enum([
        'ASC',
        'DESC'
    ]).optional().default('ASC')),
    orderBy: _zod.z.enum([
        'dueTime',
        'name'
    ]).optional().default('dueTime'),
    statuses: _zod.z.array(_zod.z.enum(Object.values(_constants.TASK_STATUSES))).optional().default([
        _constants.TASK_STATUSES.TODO
    ]),
    assignedTo: _zod.z.string().optional(),
    page: _zod.z.coerce.number().optional().default(0),
    rowsPerPage: _zod.z.coerce.number().optional().default(10)
});
encounterRelations.get('/:id/tasks', (0, _expressasynchandler.default)(async (req, res)=>{
    const { models, params } = req;
    const { Task } = models;
    const { id: encounterId } = params;
    const query = await encounterTasksQuerySchema.parseAsync(req.query);
    const { order, orderBy, assignedTo, statuses, page, rowsPerPage } = query;
    req.checkPermission('list', 'Tasking');
    const upcomingTasksTimeFrame = _config.default.tasking?.upcomingTasksTimeFrame || 8;
    const baseQueryOptions = {
        where: {
            encounterId,
            status: {
                [_sequelize.Op.in]: statuses
            },
            dueTime: {
                [_sequelize.Op.lte]: (0, _countryDateTime.toCountryDateTimeString)((0, _datefns.add)(new Date(), {
                    hours: upcomingTasksTimeFrame
                }))
            },
            taskType: {
                [_sequelize.Op.notIn]: _constants.DASHBOARD_ONLY_TASK_TYPES
            },
            ...assignedTo && {
                [_sequelize.Op.and]: (0, _sequelize.literal)(`
            EXISTS (
              SELECT 1 FROM "task_designations" AS td
              WHERE (
                "td"."designation_id" = :assignedTo
                AND "td"."task_id" = "Task"."id"
              )
            )
          `)
            }
        },
        replacements: {
            assignedTo
        }
    };
    const queryResults = await Task.findAll({
        ...baseQueryOptions,
        order: [
            [
                orderBy,
                order.toUpperCase()
            ],
            [
                'highPriority',
                'DESC'
            ],
            [
                'name',
                'ASC'
            ]
        ],
        limit: rowsPerPage,
        offset: page * rowsPerPage,
        include: [
            ...Task.getFullReferenceAssociations(),
            'parentTask'
        ]
    });
    const results = queryResults.map((x)=>x.forResponse());
    const count = await Task.count(baseQueryOptions);
    res.send({
        data: results,
        count
    });
}));
encounterRelations.get('/:id/charts/:chartSurveyId/chartInstances', (0, _charts.fetchChartInstances)({
    permissionAction: 'list'
}));
encounterRelations.delete('/:id/chartInstances/:chartInstanceResponseId', (0, _charts.deleteChartInstance)());
encounter.use(encounterRelations);

//# sourceMappingURL=encounter.js.map