"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "STEPS", {
    enumerable: true,
    get: function() {
        return STEPS;
    }
});
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _xlsx = /*#__PURE__*/ _interop_require_wildcard(require("xlsx"));
const _path = /*#__PURE__*/ _interop_require_default(require("path"));
const _fs = /*#__PURE__*/ _interop_require_default(require("fs"));
const _step = require("../step.js");
const _constants = require("@tamanu/constants");
const _utils = require("@tamanu/shared/utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
async function download(metaServerHost, artifactType, extractor, { toVersion, log }) {
    const url = `${metaServerHost}/versions/${toVersion}/artifacts`;
    log.info('Downloading translation artifacts', {
        version: toVersion,
        url
    });
    const artifactsResponse = await fetch(url, {
        headers: {
            Accept: 'application/json'
        }
    });
    if (!artifactsResponse.ok) {
        throw new Error(`Failed to fetch artifacts list: ${artifactsResponse.status} ${artifactsResponse.statusText}`);
    }
    const artifacts = await artifactsResponse.json();
    const translationsArtifact = artifacts.find((artifact)=>artifact.artifact_type === artifactType);
    if (!translationsArtifact) {
        throw new Error(`No ${artifactType} artifact found for version ${toVersion}`);
    }
    log.info(`Downloading ${artifactType} artifact`, {
        url: translationsArtifact.download_url
    });
    const translationsResponse = await fetch(translationsArtifact.download_url);
    if (!translationsResponse.ok) {
        throw new Error(`Failed to download ${artifactType} artifact: ${translationsResponse.status} ${translationsResponse.statusText}`);
    }
    return await extractor(translationsResponse);
}
// Tries each meta server in the array until one succeeds
// Returns the translations or throws an error if all fail
async function downloadFromMetaServerHosts(artifactType, extractor, stepArgs) {
    const metaServerHosts = (0, _utils.getMetaServerHosts)();
    const { log } = stepArgs;
    for (const metaServerHost of metaServerHosts){
        try {
            const rows = await download(metaServerHost, artifactType, extractor, stepArgs);
            return rows;
        } catch (error) {
            log.error(`Failed to download from meta server host: ${metaServerHost}`, {
                error
            });
        }
    }
    throw new Error('No meta server succeeded downloading the artifacts');
}
async function apply(artifactType, rows, { models, log }) {
    if (rows.length === 0) {
        throw new Error('No valid translation rows found');
    }
    log.info(`Importing ${artifactType}`, {
        count: rows.length
    });
    if (rows.length > 0) {
        await models.TranslatedString.sequelize.query(`
          INSERT INTO translated_strings (string_id, text, language)
          VALUES ${rows.map(()=>`(?, ?, '${_constants.DEFAULT_LANGUAGE_CODE}')`).join(', ')}
          ON CONFLICT (string_id, language) DO UPDATE SET text = EXCLUDED.text
        `, {
            replacements: rows.flatMap((row)=>{
                const text = row[_constants.DEFAULT_LANGUAGE_CODE] ?? row[_constants.ENGLISH_LANGUAGE_CODE] ?? row.fallback;
                return text ? [
                    row.stringId,
                    text
                ] : [];
            }),
            type: _sequelize.QueryTypes.INSERT
        });
    }
}
async function xlsxExtractor(resp) {
    const data = await resp.arrayBuffer();
    const workbook = _xlsx.read(data, {
        type: 'buffer'
    });
    const sheet = Object.values(workbook.Sheets)[0];
    if (!sheet) {
        throw new Error('No sheet found in the XLSX file');
    }
    const rows = _xlsx.utils.sheet_to_json(sheet);
    return (0, _lodash.uniqBy)(rows, (item)=>item.stringId);
}
const STEPS = [
    {
        at: _step.END,
        async check ({ serverType }) {
            // Only run on central server
            return serverType === 'central';
        },
        async run (args) {
            const zeroPatch = args.toVersion.replace(/\.(\d+)$/, '.0');
            try {
                const updateDistCjsIndexJsPath = require.resolve('@tamanu/upgrade');
                const defaultTranslationsPath = _path.default.join(updateDistCjsIndexJsPath, '..', '..', 'default-translations.json');
                args.log.info(`Loading default all translations from: ${defaultTranslationsPath}`);
                const translationRows = JSON.parse(_fs.default.readFileSync(defaultTranslationsPath, 'utf8'));
                // Add default language name and country code
                translationRows.unshift({
                    stringId: _constants.LANGUAGE_NAME_STRING_ID,
                    [_constants.DEFAULT_LANGUAGE_CODE]: _constants.ENGLISH_LANGUAGE_NAME
                });
                translationRows.unshift({
                    stringId: _constants.COUNTRY_CODE_STRING_ID,
                    [_constants.DEFAULT_LANGUAGE_CODE]: _constants.ENGLISH_COUNTRY_CODE
                });
                args.log.info('Importing new default translations', {
                    count: translationRows.length
                });
                if (translationRows.length > 0) {
                    await apply('translations', translationRows, args);
                    // Set english language name and country code to default if not present
                    const englishLanguageName = await args.models.TranslatedString.findOne({
                        where: {
                            stringId: _constants.LANGUAGE_NAME_STRING_ID,
                            language: _constants.ENGLISH_LANGUAGE_CODE
                        },
                        paranoid: false
                    });
                    if (!englishLanguageName) {
                        await args.models.TranslatedString.create({
                            stringId: _constants.LANGUAGE_NAME_STRING_ID,
                            language: _constants.ENGLISH_LANGUAGE_CODE,
                            text: _constants.ENGLISH_LANGUAGE_NAME
                        });
                    }
                    const englishCountryCode = await args.models.TranslatedString.findOne({
                        where: {
                            stringId: _constants.COUNTRY_CODE_STRING_ID,
                            language: _constants.ENGLISH_LANGUAGE_CODE
                        },
                        paranoid: false
                    });
                    if (!englishCountryCode) {
                        await args.models.TranslatedString.create({
                            stringId: _constants.COUNTRY_CODE_STRING_ID,
                            language: _constants.ENGLISH_LANGUAGE_CODE,
                            text: _constants.ENGLISH_COUNTRY_CODE
                        });
                    }
                    args.log.info('Successfully imported default translations');
                }
            } catch (error) {
                args.log.error('Failed to import default translations, you will need to manually import them', {
                    error
                });
            }
            try {
                const rows = await downloadFromMetaServerHosts('report-translations', xlsxExtractor, {
                    ...args,
                    toVersion: zeroPatch
                });
                await apply('report-translations', rows, {
                    ...args,
                    toVersion: zeroPatch
                });
                args.log.info('Successfully imported default report translations');
            } catch (error) {
                args.log.error('Failed to import default report translations, you will need to manually import them', {
                    error
                });
            }
        }
    }
];

//# sourceMappingURL=1752105049000-importDefaultTranslations.js.map