"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "importSheet", {
    enumerable: true,
    get: function() {
        return importSheet;
    }
});
const _xlsx = require("xlsx");
const _constants = require("@tamanu/constants");
const _errors = require("../errors");
const _stats = require("../stats");
const _importRows = require("../importer/importRows");
const FOREIGN_KEY_SCHEMATA = {
    CertifiableVaccine: [
        {
            field: 'vaccine',
            model: 'ReferenceData',
            types: [
                'drug'
            ]
        },
        {
            field: 'manufacturer',
            model: 'ReferenceData',
            types: [
                'manufacturer'
            ]
        }
    ],
    Department: [
        {
            field: 'facility',
            model: 'Facility'
        }
    ],
    Facility: [
        {
            field: 'catchment',
            model: 'ReferenceData',
            types: [
                'catchment'
            ]
        }
    ],
    LabTestType: [
        {
            field: 'labTestCategory',
            model: 'ReferenceData',
            types: [
                'labTestCategory'
            ]
        }
    ],
    Location: [
        {
            field: 'facility',
            model: 'Facility'
        }
    ],
    Patient: [
        {
            field: 'village',
            model: 'ReferenceData',
            types: [
                'village'
            ]
        }
    ],
    Permission: [
        {
            field: 'role',
            model: 'Role'
        }
    ],
    ScheduledVaccine: [
        {
            field: 'vaccine',
            model: 'ReferenceData',
            types: [
                'drug'
            ]
        }
    ],
    ReferenceDataRelation: [
        {
            field: 'referenceData',
            model: 'ReferenceData',
            types: _constants.REFERENCE_TYPE_VALUES
        },
        {
            field: 'referenceDataParent',
            model: 'ReferenceData',
            types: _constants.REFERENCE_TYPE_VALUES
        }
    ]
};
// https://github.com/SheetJS/sheetjs/issues/214#issuecomment-96843418
const extractHeader = (sheet)=>{
    const header = [];
    const range = _xlsx.utils.decode_range(sheet['!ref']);
    let C, R = range.s.r; /* start in the first row */ 
    /* walk every column in the range */ for(C = range.s.c; C <= range.e.c; ++C){
        const cell = sheet[_xlsx.utils.encode_cell({
            c: C,
            r: R
        })]; /* find the cell in the first row */ 
        if (cell && cell.t) header.push(_xlsx.utils.format_cell(cell));
    }
    return header;
};
async function importSheet({ errors, log, models }, { loader, sheetName, sheet, skipExisting }) {
    const stats = {};
    log.debug('Loading rows from sheet');
    let sheetHeader;
    let sheetRows;
    try {
        sheetHeader = extractHeader(sheet);
        // For drug sheets, include empty cells so facility columns with empty values are preserved
        sheetRows = sheetName === _constants.REFERENCE_TYPES.DRUG ? _xlsx.utils.sheet_to_json(sheet, {
            defval: ''
        }) : _xlsx.utils.sheet_to_json(sheet);
    } catch (err) {
        errors.push(new _errors.WorkSheetError(sheetName, 0, err));
        return stats;
    }
    if (sheetRows.length === 0) {
        log.debug('Nothing in this sheet, skipping');
        return stats;
    }
    log.debug('Preparing rows of data into table rows', {
        rows: sheetRows.length
    });
    const tableRows = [];
    const idCache = new Set();
    for (const [sheetRow, data] of sheetRows.entries()){
        const trimmed = Object.fromEntries(Object.entries(data).map(([key, value])=>[
                key.trim(),
                value
            ]));
        try {
            for (const { model, values } of (await loader(trimmed, {
                models,
                foreignKeySchemata: FOREIGN_KEY_SCHEMATA,
                header: sheetHeader,
                pushError: (message, errModel)=>{
                    errors.push(new _errors.ValidationError(sheetName, sheetRow, message));
                    if (errModel) {
                        (0, _stats.updateStat)(stats, (0, _stats.statkey)(errModel, sheetName), 'errored');
                    }
                }
            }))){
                if (!models[model]) throw new Error(`No such type of data: ${model}`);
                if (values.id && idCache.has(`${model}|${values.id}`)) {
                    errors.push(new _errors.ValidationError(sheetName, sheetRow, `duplicate id: ${values.id}`));
                    continue;
                } else {
                    idCache.add(`${model}|${values.id}`);
                }
                (0, _stats.updateStat)(stats, (0, _stats.statkey)(model, sheetName), 'created', 0);
                tableRows.push({
                    model,
                    sheetRow,
                    values
                });
            }
        } catch (err) {
            errors.push(new _errors.DataLoaderError(sheetName, sheetRow, err));
        }
    }
    await (0, _importRows.importRows)({
        errors,
        log,
        models
    }, {
        rows: tableRows,
        sheetName,
        stats,
        foreignKeySchemata: FOREIGN_KEY_SCHEMATA,
        skipExisting
    });
    return stats;
}

//# sourceMappingURL=sheet.js.map