"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "login", {
    enumerable: true,
    get: function() {
        return login;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _bcrypt = /*#__PURE__*/ _interop_require_default(require("bcrypt"));
const _jose = /*#__PURE__*/ _interop_require_wildcard(require("jose"));
const _constants = require("@tamanu/constants");
const _rolesToPermissions = require("@tamanu/shared/permissions/rolesToPermissions");
const _logging = require("@tamanu/shared/services/logging");
const _localisation = require("../localisation");
const _convertDbRecord = require("../convertDbRecord");
const _utils = require("./utils");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
const getRefreshToken = async (models, { refreshSecret, userId, deviceId })=>{
    const { RefreshToken } = models;
    const { auth: { saltRounds, refreshToken: { refreshIdLength, tokenDuration: refreshTokenDuration } }, canonicalHostName } = _config.default;
    const refreshId = await (0, _utils.getRandomBase64String)(refreshIdLength);
    const refreshTokenJwtId = (0, _utils.getRandomU32)();
    const [hashedRefreshId, refreshToken] = await Promise.all([
        _bcrypt.default.hash(refreshId, saltRounds),
        (0, _utils.buildToken)({
            userId,
            refreshId
        }, refreshSecret, {
            expiresIn: refreshTokenDuration,
            audience: _constants.JWT_TOKEN_TYPES.REFRESH,
            issuer: canonicalHostName,
            jwtid: `${refreshTokenJwtId}`
        })
    ]);
    // Extract expiry as set by jose.SignJWT
    const { exp } = _jose.decodeJwt(refreshToken);
    await RefreshToken.upsert({
        refreshId: hashedRefreshId,
        expiresAt: new Date(exp * 1000),
        userId,
        deviceId
    }, {
        where: {
            userId,
            deviceId
        }
    });
    return refreshToken;
};
const login = (0, _expressasynchandler.default)(async (req, res)=>{
    const { auth: { secret, tokenDuration, refreshToken: { secret: refreshSecret } }, canonicalHostName } = _config.default;
    const { store: { models, models: { User } }, body, settings } = req;
    const { token, user, device, internalClient, settings: userSettings } = await User.loginFromCredential({
        ...body,
        clientHeader: req.header('X-Tamanu-Client')
    }, {
        log: _logging.log,
        settings,
        tokenDuration,
        tokenIssuer: canonicalHostName,
        tokenSecret: secret
    });
    const [refreshToken, allowedFacilities, localisation, permissions, role] = await Promise.all([
        internalClient ? getRefreshToken(models, {
            refreshSecret,
            userId: user.id,
            deviceId: device?.id
        }) : undefined,
        user.allowedFacilities(),
        (0, _localisation.getLocalisation)(),
        (0, _rolesToPermissions.getPermissionsForRoles)(models, user.role),
        models.Role.findByPk(user.role)
    ]);
    // Send some additional data with login to tell the user about
    // the context they've just logged in to.
    res.send({
        token,
        refreshToken,
        user: (0, _convertDbRecord.convertFromDbRecord)((0, _utils.stripUser)(user.get({
            plain: true
        }))).data,
        permissions,
        serverType: _constants.SERVER_TYPES.CENTRAL,
        role: role?.forResponse() ?? null,
        allowedFacilities,
        localisation,
        centralHost: _config.default.canonicalHostName,
        settings: userSettings
    });
});

//# sourceMappingURL=login.js.map