"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    login: function() {
        return login;
    },
    requestLoginToken: function() {
        return requestLoginToken;
    }
});
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _zod = require("zod");
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _auth = require("@tamanu/constants/auth");
const _errors = require("@tamanu/errors");
const _utils = require("../../auth/utils");
const _PortalOneTimeTokenService = require("./PortalOneTimeTokenService");
const _replaceInTemplate = require("@tamanu/utils/replaceInTemplate");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const getOneTimeTokenEmail = async ({ email, token, settings })=>{
    const template = await settings.get('templates.patientPortalLoginEmail');
    const templateData = {
        token
    };
    const subject = (0, _replaceInTemplate.replaceInTemplate)(template.subject, templateData);
    const content = (0, _replaceInTemplate.replaceInTemplate)(template.body, templateData);
    return {
        to: email,
        from: _config.default.mailgun.from,
        subject,
        text: content
    };
};
const requestLoginSchema = _zod.z.object({
    email: _zod.z.email()
});
const requestLoginToken = (0, _expressasynchandler.default)(async (req, res)=>{
    const { store, body, emailService, settings } = req;
    const { models } = store;
    let email;
    try {
        const result = await requestLoginSchema.parseAsync(body);
        email = result.email;
    } catch (error) {
        throw new _errors.BadAuthenticationError('Invalid email address');
    }
    // Validate that the portal user exists
    const portalUser = await models.PortalUser.getForAuthByEmail(email);
    if (!portalUser) {
        _logging.log.debug('Patient portal login: Invalid email address', {
            email
        });
        // Avoid email enumeration by always returning a success response for invalid email addresses
        return res.status(200).json({
            message: 'One-time token sent successfully'
        });
    }
    // Do not issue login tokens for deceased patients
    const patient = await portalUser.getPatient();
    if (patient?.dateOfDeath) {
        _logging.log.debug('Patient portal login: Deceased patient - suppressing issuing token', {
            email,
            patientId: patient.id
        });
        return res.status(200).json({
            message: 'One-time token sent successfully'
        });
    }
    if (portalUser.status !== _constants.PORTAL_USER_STATUSES.REGISTERED) {
        throw new _errors.BadAuthenticationError('Email is not verified');
    }
    const oneTimeTokenService = new _PortalOneTimeTokenService.PortalOneTimeTokenService(models);
    const { token } = await oneTimeTokenService.createLoginToken(portalUser.id);
    // Send email with the 6-digit code
    const oneTimeTokenEmail = await getOneTimeTokenEmail({
        email,
        token,
        settings
    });
    const emailResult = await emailService.sendEmail(oneTimeTokenEmail);
    if (emailResult.status === _constants.COMMUNICATION_STATUSES.ERROR) {
        throw new Error('Failed to send email');
    }
    return res.status(200).json({
        message: 'One-time token sent successfully'
    });
});
const login = ({ secret })=>(0, _expressasynchandler.default)(async (req, res)=>{
        const { store, body } = req;
        const { canonicalHostName } = _config.default;
        const { models } = store;
        const { loginToken, email } = body;
        const portalUser = await models.PortalUser.getForAuthByEmail(email);
        const patient = await portalUser?.getPatient();
        let portalUserIdParam = portalUser?.id;
        if (!portalUser) {
            _logging.log.debug('Patient portal login: suppressing issuing token for unknown user', {
                email
            });
            // If the email is unknown, pass undefined so the service throws a generic auth error.
            portalUserIdParam = undefined;
        } else if (patient?.dateOfDeath) {
            _logging.log.debug('Patient portal login: suppressing issuing token for deceased patient', {
                email,
                patientId: patient.id,
                portalUserId: portalUser.id
            });
            // If the patient is deceased, pass undefined so the service throws a generic auth error.
            portalUserIdParam = undefined;
        }
        const oneTimeTokenService = new _PortalOneTimeTokenService.PortalOneTimeTokenService(models);
        await oneTimeTokenService.verifyAndConsume({
            token: loginToken,
            portalUserId: portalUserIdParam
        });
        const patientPortalTokenDuration = _config.default.patientPortal.tokenDuration;
        const token = await (0, _utils.buildToken)({
            portalUserId: portalUser.id
        }, secret, {
            expiresIn: patientPortalTokenDuration,
            audience: _auth.JWT_TOKEN_TYPES.PATIENT_PORTAL_ACCESS,
            issuer: canonicalHostName
        });
        return res.status(200).json({
            token
        });
    });

//# sourceMappingURL=login.js.map