"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "reportCommand", {
    enumerable: true,
    get: function() {
        return reportCommand;
    }
});
const _commander = require("commander");
const _path = /*#__PURE__*/ _interop_require_default(require("path"));
const _logging = require("@tamanu/shared/services/logging");
const _reports = require("@tamanu/shared/reports");
const _constants = require("@tamanu/constants");
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _EmailService = require("../services/EmailService");
const _ReportRunner = require("../report/ReportRunner");
const _database = require("../database");
const _env = require("../env");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const REPORT_HEAP_INTERVAL_MS = 1000;
const validateReportId = async (reportId, models)=>{
    const dbDefinedReportModule = await models.ReportDefinitionVersion.findByPk(reportId);
    if (dbDefinedReportModule) {
        return true;
    }
    const validNames = _reports.REPORT_DEFINITIONS.map((d)=>d.id);
    if (!validNames.includes(reportId)) {
        const nameOutput = validNames.map((n)=>`\n  ${n}`).join('');
        throw new Error(`invalid name '${reportId}', must be one of: ${nameOutput} \n (hint - supply name with --reportId <reportId>)`);
    }
    return true;
};
async function report(options) {
    if (options.heap) {
        setInterval(()=>{
            const used = process.memoryUsage().heapUsed / 1024 / 1024;
            // eslint-disable-next-line no-console
            console.log(`Heap: ${used} MiB`);
        }, REPORT_HEAP_INTERVAL_MS);
    }
    const store = await (0, _database.initDatabase)({
        testMode: false
    });
    const reportSchemaStores = _config.default.db.reportSchemas?.enabled ? await (0, _database.initReporting)() : null;
    (0, _env.setupEnv)();
    try {
        const { reportId, parameters, recipients, userId, format, sleepAfterReport } = options;
        await validateReportId(reportId, store.models);
        let reportParameters = {};
        let reportRecipients = {};
        try {
            reportParameters = JSON.parse(parameters);
        } catch (error) {
            _logging.log.warn(`Failed to parse parameters ${error}`);
        }
        try {
            reportRecipients = JSON.parse(recipients);
        } catch (error) {
            // Backwards compatibility: support previous syntax of plain string
            _logging.log.warn(error);
            reportRecipients = {
                email: recipients.split(',')
            };
        }
        const emailService = new _EmailService.EmailService();
        const reportRunner = new _ReportRunner.ReportRunner(reportId, reportParameters, reportRecipients, store, reportSchemaStores, emailService, userId, format, sleepAfterReport);
        _logging.log.info(`Running report "${reportId}" with parameters "${parameters}", recipients "${recipients}" and userId ${userId}`);
        await reportRunner.run();
    } catch (error) {
        // Send error message back to parent process
        process.stderr.write(`Report failed: ${error.message}\n`);
        process.exit(1);
    }
    process.exit(0);
}
const reportCommand = new _commander.Command('report').description('Generate a report').option('--reportId <string>', 'id of the report') // validated in function
.option('--heap', `Report heap usage every ${REPORT_HEAP_INTERVAL_MS}ms`, false).requiredOption('-r, --recipients <json|csv>', 'JSON recipients or comma-separated list of emails', // {"local":[{"format": "csv","path":"./tamanu-reports"}]}
JSON.stringify({
    local: [
        {
            format: 'csv',
            path: _path.default.join(process.cwd(), 'tamanu-reports')
        },
        {
            format: 'xlsx',
            path: _path.default.join(process.cwd(), 'tamanu-reports')
        }
    ]
})).option('-p, --parameters <json>', 'JSON parameters').option('-u, --userId <string>', 'Requested by userId').option('-f, --format <string>', 'Export format (xslx or csv)', _constants.REPORT_EXPORT_FORMATS.XLSX).option('-s, --sleepAfterReport <json>', 'Sleep thread for `duration` if report takes longer than `ifRunAtLeast`').action(report);

//# sourceMappingURL=report.js.map