"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "IPSRequestProcessor", {
    enumerable: true,
    get: function() {
        return IPSRequestProcessor;
    }
});
const _config = /*#__PURE__*/ _interop_require_default(require("config"));
const _path = /*#__PURE__*/ _interop_require_default(require("path"));
const _jose = /*#__PURE__*/ _interop_require_wildcard(require("jose"));
const _clients3 = /*#__PURE__*/ _interop_require_wildcard(require("@aws-sdk/client-s3"));
const _encodings = require("@tamanu/utils/encodings");
const _constants = require("@tamanu/constants");
const _logging = require("@tamanu/shared/services/logging");
const _tasks = require("@tamanu/shared/tasks");
const _utils = require("@tamanu/shared/utils");
const _utils1 = require("../../auth/utils");
const _bundleGenerator = require("../../hl7fhir/materialised/patientSummary/bundleGenerator");
const _helpers = require("./helpers");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
function _getRequireWildcardCache(nodeInterop) {
    if (typeof WeakMap !== "function") return null;
    var cacheBabelInterop = new WeakMap();
    var cacheNodeInterop = new WeakMap();
    return (_getRequireWildcardCache = function(nodeInterop) {
        return nodeInterop ? cacheNodeInterop : cacheBabelInterop;
    })(nodeInterop);
}
function _interop_require_wildcard(obj, nodeInterop) {
    if (!nodeInterop && obj && obj.__esModule) {
        return obj;
    }
    if (obj === null || typeof obj !== "object" && typeof obj !== "function") {
        return {
            default: obj
        };
    }
    var cache = _getRequireWildcardCache(nodeInterop);
    if (cache && cache.has(obj)) {
        return cache.get(obj);
    }
    var newObj = {
        __proto__: null
    };
    var hasPropertyDescriptor = Object.defineProperty && Object.getOwnPropertyDescriptor;
    for(var key in obj){
        if (key !== "default" && Object.prototype.hasOwnProperty.call(obj, key)) {
            var desc = hasPropertyDescriptor ? Object.getOwnPropertyDescriptor(obj, key) : null;
            if (desc && (desc.get || desc.set)) {
                Object.defineProperty(newObj, key, desc);
            } else {
                newObj[key] = obj[key];
            }
        }
    }
    newObj.default = obj;
    if (cache) {
        cache.set(obj, newObj);
    }
    return newObj;
}
// SHL flag reference: https://docs.smarthealthit.org/smart-health-links/spec/#construct-a-shlink-payload
const SHL_FLAG_LONGTERM = 'L';
const SHL_FLAG_SINGLEFILE = 'U';
let IPSRequestProcessor = class IPSRequestProcessor extends _tasks.ScheduledTask {
    getName() {
        return 'IPSRequestProcessor';
    }
    async countQueue() {
        return this.context.store.models.IPSRequest.count({
            where: {
                status: _constants.IPS_REQUEST_STATUSES.QUEUED
            }
        });
    }
    async run() {
        const { models } = this.context.store;
        const { FhirPatient, IPSRequest, PatientCommunication, User } = models;
        const queuedNotifications = await IPSRequest.findAll({
            where: {
                status: _constants.IPS_REQUEST_STATUSES.QUEUED
            },
            order: [
                [
                    'createdAt',
                    'ASC'
                ]
            ],
            limit: this.config.limit
        });
        let processed = 0;
        for (const notification of queuedNotifications){
            try {
                const { patientId, createdBy, email } = notification;
                const [fhirPatient, user] = await Promise.all([
                    FhirPatient.findOne({
                        where: {
                            upstreamId: patientId
                        }
                    }),
                    User.findByPk(createdBy)
                ]);
                if (!fhirPatient) {
                    throw new Error(`No FHIR patient with patient id ${patientId}`);
                }
                const { patient, bundle: ipsJSON } = await (0, _bundleGenerator.generateIPSBundle)(fhirPatient.id, user, models);
                const sublog = _logging.log.child({
                    id: notification.id,
                    patient: patientId,
                    fhirPatient: patient.id,
                    createdBy,
                    email
                });
                sublog.info('Processing IPS request');
                const now = new Date();
                const { region, bucketName, jsonBucketPath, viewerBucketPath, publicUrl: s3PublicUrl } = _config.default.s3.ips;
                if (!jsonBucketPath) {
                    throw new Error(`jsonBucketPath must be set, e.g. 'au'`);
                }
                const filePath = `${jsonBucketPath}/${await (0, _utils1.getRandomBase64String)(32, 'base64url')}_manifest`;
                // CREATE PAYLOAD
                const payload = {
                    url: `${s3PublicUrl}/${filePath}`,
                    key: await (0, _utils1.getRandomBase64String)(32, 'base64url'),
                    flag: `${SHL_FLAG_LONGTERM}${SHL_FLAG_SINGLEFILE}`,
                    label: `${patient.displayName} International Patient Summary generated @ ${now.toLocaleString()}`
                };
                // ENCRYPT IPS BUNDLE
                const encrypted = await new _jose.CompactEncrypt(new TextEncoder().encode(JSON.stringify(ipsJSON))).setProtectedHeader({
                    alg: 'dir',
                    enc: 'A256GCM',
                    cty: 'application/fhir+json'
                }).encrypt(_jose.base64url.decode(payload.key));
                // SAVE BUNDLE TO S3
                try {
                    const client = new _clients3.S3({
                        region
                    });
                    await client.send(new _clients3.PutObjectCommand({
                        Bucket: bucketName,
                        Key: filePath,
                        Body: encrypted,
                        ContentType: 'application/jose'
                    }));
                } catch (err) {
                    throw new Error(`There was an error uploading to S3, ${err}`);
                }
                const baseUrl = `${s3PublicUrl}/${viewerBucketPath}`;
                const fullUrl = `${baseUrl}#shlink:/${(0, _encodings.base64UrlEncode)(JSON.stringify(payload))}`;
                // GENERATE QR
                const qrCodeFileName = `International Patient Summary for ${patient.displayName}.png`;
                const folder = await (0, _utils.tmpdir)();
                const qrCodeFilePath = _path.default.join(folder, qrCodeFileName);
                await (0, _helpers.QRCodeToFileAsync)(qrCodeFilePath, fullUrl, {
                    type: 'png'
                });
                // SEND EMAIL
                const { subject, bodyText } = _config.default.integrations.ips.email;
                const content = `
          ${bodyText} \n  Alternatively, use the following link ${fullUrl} \n
          Do not respond to this email.
        `;
                sublog.debug('Creating communication record');
                // build the email notification
                const comm = await PatientCommunication.create({
                    type: _constants.PATIENT_COMMUNICATION_TYPES.CERTIFICATE,
                    channel: _constants.PATIENT_COMMUNICATION_CHANNELS.EMAIL,
                    subject,
                    content,
                    status: _constants.COMMUNICATION_STATUSES.QUEUED,
                    patientId,
                    destination: email,
                    attachment: qrCodeFilePath
                });
                sublog.info('Done processing IPS request', {
                    emailId: comm.id
                });
                processed += 1;
                await notification.update({
                    status: _constants.IPS_REQUEST_STATUSES.PROCESSED
                });
            } catch (error) {
                _logging.log.error('Failed to process IPS request', {
                    id: notification.id,
                    error
                });
                await notification.update({
                    status: _constants.IPS_REQUEST_STATUSES.ERROR,
                    error: error.message
                });
            }
        }
        _logging.log.info('Done with IPS request task', {
            attempted: queuedNotifications.length,
            processed
        });
    }
    constructor(context){
        const conf = _config.default.schedules.IPSRequestProcessor;
        const { schedule, enabled } = conf;
        super(schedule, _logging.log, undefined, enabled);
        this.config = conf;
        this.context = context;
    }
};

//# sourceMappingURL=index.js.map