"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
function _export(target, all) {
    for(var name in all)Object.defineProperty(target, name, {
        enumerable: true,
        get: all[name]
    });
}
_export(exports, {
    compositeToSql: function() {
        return compositeToSql;
    },
    setupQuote: function() {
        return setupQuote;
    }
});
const _sequelize = require("sequelize");
const _datefns = require("date-fns");
const _sequelizeType = require("./sequelizeType");
let QUERY_GENERATOR;
function setupQuote(sequelizeInstance) {
    QUERY_GENERATOR = new WeakRef(sequelizeInstance.connectionManager.dialect.queryGenerator);
}
/**
 * The quote function from the Sequelize dialect QueryGenerator.
 *
 * This must be set up via the setupQuote() function before use.
 */ function quote(value) {
    return QUERY_GENERATOR.deref().quote(value);
}
function compositeToSql(fieldSet) {
    return `(${fieldSet.map(compositeField).join(',')})`;
}
function compositeField(field) {
    switch(typeof field){
        case 'number':
        case 'boolean':
        case 'bigint':
            return field.toString();
        case 'undefined':
            return '';
        case 'string':
        case 'symbol':
            return compositeString(field);
        case 'function':
            return compositeField(field());
        case 'object':
            {
                if (field === null) {
                    // > A completely empty field value (no characters at all between the commas or parentheses)
                    // > represents a NULL.
                    return '';
                }
                if (Array.isArray(field)) {
                    return compositeString(arrayToSql(field));
                }
                if (field instanceof _sequelizeType.Composite) {
                    return compositeString(compositeToSql(field.sqlFields()));
                }
                if (field instanceof Date) {
                    return compositeString((0, _datefns.formatISO9075)(field));
                }
                if (field instanceof _sequelize.Utils.Fn || field instanceof _sequelize.Utils.Col || field instanceof _sequelize.Utils.Literal || field instanceof _sequelize.Utils.Fn || field instanceof _sequelize.Utils.Json || field instanceof _sequelize.Utils.Cast) {
                    return compositeString(quote(field));
                }
                throw new Error(`unsupported type to stringify to composite: ${field.constructor}`);
            }
        default:
            throw new Error(`unknown typeof return value: ${typeof field}`);
    }
}
function compositeString(string) {
    // > When writing a composite value you can write double quotes around any individual field value.
    // > You *must* do so if the field value would otherwise confuse the composite-value parser. In
    // > particular, fields containing parentheses, commas, double quotes, or backslashes must be
    // > double-quoted. To put a double quote or backslash in a quoted composite field value, precede
    // > it with a backslash.
    // Let's do this simply by double-quoting everything.
    return `"${string.replaceAll(/(\\|")/g, (c)=>`\\${c}`)}"`;
}
function arrayToSql(arr) {
    return `{${arr.map((a)=>compositeField(a)).join(',')}}`;
}

//# sourceMappingURL=stringifier.js.map