"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "PharmacyOrderPrescription", {
    enumerable: true,
    get: function() {
        return PharmacyOrderPrescription;
    }
});
const _sequelize = require("sequelize");
const _constants = require("@tamanu/constants");
const _generateDisplayId = require("@tamanu/utils/generateDisplayId");
const _Model = require("./Model");
const _sync = require("../sync");
let PharmacyOrderPrescription = class PharmacyOrderPrescription extends _Model.Model {
    static initModel({ primaryKey, ...options }, models) {
        super.init({
            id: primaryKey,
            displayId: {
                type: _sequelize.DataTypes.STRING,
                allowNull: true,
                defaultValue () {
                    return (0, _generateDisplayId.generateDisplayId)();
                }
            },
            quantity: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: false
            },
            repeats: {
                type: _sequelize.DataTypes.INTEGER,
                allowNull: true
            },
            isCompleted: {
                type: _sequelize.DataTypes.BOOLEAN,
                allowNull: false,
                defaultValue: false
            }
        }, {
            ...options,
            syncDirection: _constants.SYNC_DIRECTIONS.BIDIRECTIONAL,
            hooks: {
                async afterDestroy (pharmacyOrderPrescription, opts) {
                    const pharmacyOrder = await models.PharmacyOrder.findByPk(pharmacyOrderPrescription.pharmacyOrderId, {
                        include: [
                            {
                                association: 'pharmacyOrderPrescriptions'
                            }
                        ],
                        transaction: opts.transaction
                    });
                    if (pharmacyOrder && (!pharmacyOrder?.pharmacyOrderPrescriptions || !pharmacyOrder?.pharmacyOrderPrescriptions.length)) {
                        await pharmacyOrder.destroy({
                            transaction: opts.transaction
                        });
                    }
                }
            }
        });
    }
    static initRelations(models) {
        this.belongsTo(models.PharmacyOrder, {
            foreignKey: 'pharmacyOrderId',
            as: 'pharmacyOrder'
        });
        this.belongsTo(models.Prescription, {
            foreignKey: 'prescriptionId',
            as: 'prescription'
        });
        this.belongsTo(models.Prescription, {
            foreignKey: 'ongoingPrescriptionId',
            as: 'ongoingPrescription'
        });
        this.hasMany(models.MedicationDispense, {
            foreignKey: 'pharmacyOrderPrescriptionId',
            as: 'medicationDispenses'
        });
    }
    getRemainingRepeats(extraDispenses = 0) {
        // No repeats will be consumed by an INPATIENT medication request.
        if (!this.pharmacyOrder?.isDischargePrescription) {
            return 0;
        }
        // The remaining repeats for OUTPATIENT medication requests is the number of repeats minus the number of dispenses.
        const repeats = this.repeats || 0;
        const dispenseCount = (this.medicationDispenses || []).length + extraDispenses;
        // we subtract 1 from the dispense count because the first dispense is not counted as a repeat
        return Math.max(0, repeats - Math.max(0, dispenseCount - 1));
    }
    static getListReferenceAssociations() {
        return [
            'pharmacyOrder',
            'prescription'
        ];
    }
    static buildPatientSyncFilter(patientCount, markedForSyncPatientsTable) {
        if (patientCount === 0) {
            return null;
        }
        return (0, _sync.buildEncounterLinkedSyncFilter)([
            this.tableName,
            'pharmacy_orders',
            'encounters'
        ], markedForSyncPatientsTable);
    }
    static async buildSyncLookupQueryDetails() {
        return {
            select: await (0, _sync.buildEncounterPatientIdSelect)(this),
            joins: (0, _sync.buildEncounterLinkedSyncFilterJoins)([
                this.tableName,
                'pharmacy_orders',
                'encounters'
            ])
        };
    }
};

//# sourceMappingURL=PharmacyOrderPrescription.js.map