import { SYNC_DIRECTIONS, SETTING_KEYS, LOGIN_ATTEMPT_OUTCOMES } from '@tamanu/constants';
import { Model } from './Model';
import { DataTypes, Sequelize, Op } from 'sequelize';
import { log } from '@tamanu/shared/services/logging';
export class UserLoginAttempt extends Model {
    static initModel(options) {
        super.init({
            id: {
                type: DataTypes.UUID,
                allowNull: false,
                primaryKey: true,
                defaultValue: Sequelize.fn('gen_random_uuid')
            },
            outcome: {
                type: DataTypes.TEXT,
                allowNull: false
            },
            deviceId: {
                type: DataTypes.TEXT,
                allowNull: true
            }
        }, {
            ...options,
            syncDirection: SYNC_DIRECTIONS.PUSH_TO_CENTRAL
        });
    }
    static initRelations(models) {
        this.belongsTo(models.User, {
            foreignKey: 'userId',
            as: 'user'
        });
    }
    // If the deviceId is not found in the devices table, we will use null
    // for the deviceId to avoid credential stuffing from unregistered devices.
    static async getDeviceIdToUse(deviceId) {
        const device = await this.sequelize.models.Device.findByPk(deviceId);
        return device ? deviceId : null;
    }
    static async checkIsUserLockedOut({ settings, userId, deviceId = '' }) {
        const { lockoutDuration } = await settings.get(SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const deviceIdToUse = await this.getDeviceIdToUse(deviceId);
        const lockedAttempt = await this.findOne({
            where: {
                userId,
                deviceId: deviceIdToUse,
                outcome: LOGIN_ATTEMPT_OUTCOMES.LOCKED,
                createdAt: {
                    [Op.gte]: Sequelize.literal("CURRENT_TIMESTAMP - $lockoutDuration * interval '1 minute'")
                }
            },
            order: [
                [
                    'createdAt',
                    'DESC'
                ]
            ],
            bind: {
                lockoutDuration
            }
        });
        const createdAt = lockedAttempt?.createdAt?.getTime() ?? 0;
        const remainingLockout = lockoutDuration * 60 - Math.floor((Date.now() - createdAt) / 1000);
        return {
            isUserLockedOut: !!lockedAttempt,
            remainingLockout: Math.max(0, remainingLockout)
        };
    }
    static async createFailedLoginAttempt({ settings, userId, deviceId = '' }) {
        const { lockoutThreshold, observationWindow, lockoutDuration } = await settings.get(SETTING_KEYS.SECURITY_LOGIN_ATTEMPTS);
        const deviceIdToUse = await this.getDeviceIdToUse(deviceId);
        const failedAttempts = await this.count({
            where: {
                userId,
                deviceId: deviceIdToUse,
                outcome: LOGIN_ATTEMPT_OUTCOMES.FAILED,
                createdAt: {
                    [Op.gte]: Sequelize.literal("CURRENT_TIMESTAMP - $observationWindow * interval '1 minute'")
                }
            },
            // @ts-ignore - sequelize doesn't know bind works in count
            bind: {
                observationWindow
            }
        });
        // We need to add 1 to the failed attempts because the current attempt is not included in the count
        const outcome = failedAttempts + 1 >= lockoutThreshold ? LOGIN_ATTEMPT_OUTCOMES.LOCKED : LOGIN_ATTEMPT_OUTCOMES.FAILED;
        let loginAttempt = null;
        try {
            loginAttempt = await this.create({
                userId,
                deviceId: deviceIdToUse,
                outcome
            });
        } catch (error) {
            log.error('Error creating failed login attempt', error);
        }
        return {
            loginAttempt,
            lockoutDuration: lockoutDuration * 60,
            remainingAttempts: Math.max(0, lockoutThreshold - failedAttempts - 1)
        };
    }
}

//# sourceMappingURL=UserLoginAttempt.js.map