"use strict";
Object.defineProperty(exports, "__esModule", {
    value: true
});
Object.defineProperty(exports, "appointments", {
    enumerable: true,
    get: function() {
        return appointments;
    }
});
const _datefns = require("date-fns");
const _express = /*#__PURE__*/ _interop_require_default(require("express"));
const _expressasynchandler = /*#__PURE__*/ _interop_require_default(require("express-async-handler"));
const _sequelize = require("sequelize");
const _lodash = require("lodash");
const _zod = require("zod");
const _constants = require("@tamanu/constants");
const _errors = require("@tamanu/errors");
const _replaceInTemplate = require("@tamanu/utils/replaceInTemplate");
const _dateTime = require("@tamanu/utils/dateTime");
const _query = require("../../utils/query");
function _interop_require_default(obj) {
    return obj && obj.__esModule ? obj : {
        default: obj
    };
}
const appointments = _express.default.Router();
const reorderAppointmentSchema = _zod.z.object({
    appointments: _zod.z.array(_zod.z.object({
        id: _zod.z.string(),
        startTime: _dateTime.datetimeCustomValidation,
        endTime: _dateTime.datetimeCustomValidation
    })).min(1)
});
appointments.put('/reorder-location-bookings', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Appointment');
    const { models } = req;
    const { Appointment } = models;
    const body = await reorderAppointmentSchema.parseAsync(req.body);
    const { appointments } = body;
    const appointmentIds = appointments.map((appointment)=>appointment.id);
    const appointmentsToReorder = await Appointment.findAll({
        where: {
            id: {
                [_sequelize.Op.in]: appointmentIds
            }
        },
        include: [
            'locationGroup',
            'location'
        ]
    });
    if (appointmentIds.length !== appointmentsToReorder.length || appointmentsToReorder.some((appointment)=>!appointmentIds.includes(appointment.id))) {
        throw new _errors.NotFoundError('Some appointments not found');
    }
    const locationId = appointmentsToReorder[0].locationId;
    if (appointmentsToReorder.some((appointment)=>appointment.locationId !== locationId)) {
        throw new _errors.InvalidOperationError('All appointments must be in the same location');
    }
    if (appointmentsToReorder.some((appointment)=>appointment.status === _constants.APPOINTMENT_STATUSES.CANCELLED)) {
        throw new _errors.InvalidOperationError('Some appointments are cancelled');
    }
    const startTime = appointmentsToReorder[0].startTime;
    if (appointmentsToReorder.some((appointment)=>!(0, _datefns.isSameDay)(new Date(appointment.startTime), new Date(startTime)) || !(0, _datefns.isSameDay)(new Date(appointment.endTime), new Date(startTime)))) {
        throw new _errors.InvalidOperationError('All appointments must be in the same date');
    }
    await Appointment.sequelize.transaction(async ()=>{
        for (const appointment of appointmentsToReorder){
            await appointment.update({
                startTime: appointments.find((a)=>a.id === appointment.id).startTime,
                endTime: appointments.find((a)=>a.id === appointment.id).endTime
            });
        }
    });
    res.status(200).send({
        success: true
    });
}));
/**
 * @param {string} intervalStart Some valid PostgreSQL Date/Time input.
 * @param {string} intervalEnd Some valid PostgreSQL Date/Time input.
 * @see https://www.postgresql.org/docs/current/datatype-datetime.html#DATATYPE-DATETIME-INPUT
 */ const buildTimeQuery = (intervalStart, intervalEnd)=>{
    const whereClause = (0, _sequelize.literal)('("Appointment"."start_time"::TIMESTAMP, "Appointment"."end_time"::TIMESTAMP) OVERLAPS ($apptTimeQueryStart, $apptTimeQueryEnd)');
    const bindParams = {
        apptTimeQueryStart: `'${intervalStart}'`,
        apptTimeQueryEnd: `'${intervalEnd}'`
    };
    return [
        whereClause,
        bindParams
    ];
};
const sendAppointmentReminder = async ({ appointmentId, email, facilityId, models, settings, transaction })=>{
    const { Appointment, Facility, PatientCommunication } = models;
    // Fetch appointment relations
    const [appointment, facility] = await Promise.all([
        Appointment.findByPk(appointmentId, {
            include: [
                'patient',
                'clinician',
                'locationGroup',
                {
                    association: 'location',
                    include: [
                        'locationGroup'
                    ]
                }
            ],
            transaction
        }),
        Facility.findByPk(facilityId, {
            transaction
        })
    ]);
    const { patient, clinician, locationId } = appointment;
    const isLocationBooking = !!locationId;
    const templateKeySuffix = isLocationBooking ? 'locationBooking' : 'outpatientAppointment';
    const appointmentConfirmationTemplate = await settings[facilityId].get(`templates.appointmentConfirmation.${templateKeySuffix}`);
    const start = new Date(appointment.startTime);
    const locationName = isLocationBooking ? appointment.location?.locationGroup?.name || '' : appointment.locationGroup?.name || '';
    const content = (0, _replaceInTemplate.replaceInTemplate)(appointmentConfirmationTemplate.body, {
        firstName: patient.firstName,
        lastName: patient.lastName,
        facilityName: facility.name,
        startDate: (0, _datefns.format)(start, 'PPPP'),
        startTime: (0, _datefns.format)(start, 'p'),
        locationName,
        clinicianName: clinician?.displayName ? `\nClinician: ${clinician.displayName}` : ''
    });
    const communicationType = isLocationBooking ? _constants.PATIENT_COMMUNICATION_TYPES.BOOKING_CONFIRMATION : _constants.PATIENT_COMMUNICATION_TYPES.APPOINTMENT_CONFIRMATION;
    return await PatientCommunication.create({
        type: communicationType,
        channel: _constants.PATIENT_COMMUNICATION_CHANNELS.EMAIL,
        status: _constants.COMMUNICATION_STATUSES.QUEUED,
        destination: email,
        subject: appointmentConfirmationTemplate.subject,
        content,
        patientId: patient.id
    }, {
        transaction
    });
};
appointments.post('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Appointment');
    const { models, db, body: { facilityId, schedule: scheduleData, ...appointmentData }, settings } = req;
    const { Appointment, PatientFacility } = models;
    const result = await db.transaction(async (transaction)=>{
        const appointment = scheduleData ? (await Appointment.createWithSchedule({
            settings: settings[facilityId],
            appointmentData,
            scheduleData
        })).firstAppointment : await Appointment.create(appointmentData, {
            transaction
        });
        await PatientFacility.findOrCreate({
            where: {
                patientId: appointment.patientId,
                facilityId
            },
            transaction
        });
        const { email } = appointmentData;
        if (email) {
            await sendAppointmentReminder({
                appointmentId: appointment.id,
                email,
                facilityId,
                models,
                settings,
                transaction
            });
        }
        return appointment;
    });
    res.status(201).send(result);
}));
appointments.post('/emailReminder', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Appointment');
    const { models, body: { facilityId, appointmentId, email }, settings } = req;
    const response = await sendAppointmentReminder({
        appointmentId,
        email,
        facilityId,
        models,
        settings
    });
    res.status(200).send(response);
}));
appointments.put('/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Appointment');
    const { models, body, params, settings } = req;
    const { schedule: scheduleData, facilityId, modifyMode, ...appointmentData } = body;
    const { id } = params;
    const { Appointment } = models;
    const appointment = await Appointment.findByPk(id);
    if (!appointment) {
        throw new _errors.NotFoundError();
    }
    const result = await req.db.transaction(async ()=>{
        if (modifyMode === _constants.MODIFY_REPEATING_APPOINTMENT_MODE.THIS_AND_FUTURE_APPOINTMENTS) {
            const existingSchedule = await appointment.getSchedule();
            if (!existingSchedule) {
                throw new Error('Cannot update future appointments for a non-recurring appointment');
            }
            if (existingSchedule.isDifferentFromSchedule(scheduleData) || appointmentData.startTime !== appointment.startTime || appointmentData.endTime !== appointment.endTime) {
                // If the appointment schedule has been modified, we need to regenerate the schedule from the updated appointment.
                // To do this we cancel this and all future appointments and mark existing schedule as ended
                await existingSchedule.endAtAppointment(appointment);
                if (appointmentData.status !== _constants.APPOINTMENT_STATUSES.CANCELLED) {
                    // Then if not cancelling the repeating appointments we generate a new schedule starting with the updated appointment
                    const { schedule } = await Appointment.createWithSchedule({
                        settings: settings[facilityId],
                        appointmentData: (0, _lodash.omit)(appointmentData, 'id'),
                        scheduleData: (0, _lodash.omit)(scheduleData, 'id')
                    });
                    return {
                        schedule
                    };
                }
            } else {
                // No scheduleData or appointment time change, so this is a simple change that doesn't require deleting and regenerating future appointments
                await existingSchedule.modifyFromAppointment(appointment, // When modifying all future appointments we strip startTime, and endTime
                // in order to preserve the incremental time difference between appointments
                (0, _lodash.omit)(appointmentData, 'id', 'startTime', 'endTime'));
            }
        } else {
            await appointment.update(appointmentData);
        }
        await appointment.reload();
        return appointment;
    });
    res.status(200).send(result);
}));
const isStringOrArray = (obj)=>typeof obj === 'string' || Array.isArray(obj);
const CONTAINS_SEARCHABLE_FIELDS = [
    'patient.first_name',
    'patient.last_name',
    'patient.display_id'
];
const EXACT_MATCH_SEARCHABLE_FIELDS = [
    'startTime',
    'endTime',
    'appointmentTypeId',
    'bookingTypeId',
    'status',
    'clinicianId',
    'locationId',
    'locationGroupId',
    'patientId'
];
const ALL_SEARCHABLE_FIELDS = [
    ...CONTAINS_SEARCHABLE_FIELDS,
    ...EXACT_MATCH_SEARCHABLE_FIELDS
];
const sortKeys = {
    patientName: _sequelize.Sequelize.fn('concat', _sequelize.Sequelize.col('patient.first_name'), ' ', _sequelize.Sequelize.col('patient.last_name')),
    displayId: _sequelize.Sequelize.col('patient.display_id'),
    sex: _sequelize.Sequelize.col('patient.sex'),
    dateOfBirth: _sequelize.Sequelize.col('patient.date_of_birth'),
    location: _sequelize.Sequelize.col('location.name'),
    locationGroup: _sequelize.Sequelize.col('location_groups.name'),
    clinician: _sequelize.Sequelize.col('clinician.display_name'),
    bookingType: _sequelize.Sequelize.col('bookingType.name'),
    appointmentType: _sequelize.Sequelize.col('appointmentType.name'),
    outpatientAppointmentArea: _sequelize.Sequelize.col('locationGroup.name'),
    bookingArea: _sequelize.Sequelize.col('location.locationGroup.name')
};
const buildPatientNameOrIdQuery = (patientNameOrId)=>{
    if (!patientNameOrId) return null;
    const ilikeClause = {
        [_sequelize.Op.iLike]: `%${(0, _query.escapePatternWildcard)(patientNameOrId)}%`
    };
    return {
        [_sequelize.Op.or]: [
            _sequelize.Sequelize.where(_sequelize.Sequelize.fn('concat', _sequelize.Sequelize.col('patient.first_name'), ' ', _sequelize.Sequelize.col('patient.last_name')), ilikeClause),
            {
                '$patient.display_id$': ilikeClause
            }
        ]
    };
};
appointments.get('/$', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkListOrReadPermission('Appointment');
    const { models: { Appointment }, query: { /**
         * Midnight today
         * @see https://www.postgresql.org/docs/current/datatype-datetime.html#DATATYPE-DATETIME-SPECIAL-VALUES
         */ after = 'today', before = 'infinity', facilityId, rowsPerPage = 10, page = 0, all = false, order = 'ASC', orderBy = 'startTime', patientNameOrId, includeCancelled = false, view, ...queries } } = req;
    const [timeQueryWhereClause, timeQueryBindParams] = buildTimeQuery(after, before);
    const cancelledStatusWhereClause = includeCancelled ? null : {
        status: {
            [_sequelize.Op.not]: _constants.APPOINTMENT_STATUSES.CANCELLED
        }
    };
    const facilityIdField = isStringOrArray(queries.locationGroupId) ? '$locationGroup.facility_id$' : '$location.facility_id$';
    const facilityIdWhereClause = facilityId ? {
        [facilityIdField]: facilityId
    } : null;
    const isBeforeScheduleUntilDateWhereClause = {
        [_sequelize.Op.or]: [
            {
                scheduleId: null
            },
            (0, _sequelize.literal)(`
        ("schedule"."until_date" IS NULL OR "schedule"."until_date"::timestamp + interval '1 day' - interval '1 second' >= start_time::timestamp)
      `)
        ]
    };
    const bookableWhereClause = [
        _constants.LOCATION_BOOKABLE_VIEW.DAILY,
        _constants.LOCATION_BOOKABLE_VIEW.WEEKLY
    ].includes(view) ? {
        '$location.locationGroup.is_bookable$': {
            [_sequelize.Op.in]: [
                _constants.LOCATION_BOOKABLE_VIEW.ALL,
                view
            ]
        }
    } : null;
    const filters = Object.entries(queries).reduce((_filters, [queryField, queryValue])=>{
        if (!ALL_SEARCHABLE_FIELDS.includes(queryField) || !isStringOrArray(queryValue)) {
            return _filters;
        }
        const column = queryField.includes('.') // querying on a joined table (associations)
         ? `$${queryField}$` : queryField;
        let comparison;
        if (queryValue === '' || queryValue.length === 0) {
            comparison = {
                [_sequelize.Op.not]: null
            };
        } else if (typeof queryValue === 'string') {
            comparison = EXACT_MATCH_SEARCHABLE_FIELDS.includes(queryField) ? {
                [_sequelize.Op.eq]: queryValue
            } : {
                [_sequelize.Op.iLike]: `%${(0, _query.escapePatternWildcard)(queryValue)}%`
            };
        } else {
            comparison = {
                [_sequelize.Op.in]: queryValue
            };
        }
        _filters.push({
            [column]: comparison
        });
        return _filters;
    }, []);
    const { rows, count } = await Appointment.findAndCountAll({
        limit: all ? undefined : rowsPerPage,
        offset: all ? undefined : page * rowsPerPage,
        order: [
            [
                sortKeys[orderBy] || orderBy,
                order
            ]
        ],
        where: {
            [_sequelize.Op.and]: [
                facilityIdWhereClause,
                timeQueryWhereClause,
                cancelledStatusWhereClause,
                isBeforeScheduleUntilDateWhereClause,
                buildPatientNameOrIdQuery(patientNameOrId),
                bookableWhereClause,
                ...filters
            ]
        },
        include: Appointment.getListReferenceAssociations(),
        bind: timeQueryBindParams
    });
    res.send({
        count,
        data: rows
    });
}));
appointments.post('/locationBooking', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('create', 'Appointment');
    const { models, body, settings } = req;
    const { startTime, endTime, locationId, patientId, procedureTypeIds, email } = body;
    const { Appointment, PatientFacility, Location } = models;
    try {
        const result = await Appointment.sequelize.transaction(async (transaction)=>{
            const [timeQueryWhereClause, timeQueryBindParams] = buildTimeQuery(startTime, endTime);
            const conflictCount = await Appointment.count({
                where: {
                    [_sequelize.Op.and]: [
                        {
                            locationId
                        },
                        {
                            status: {
                                [_sequelize.Op.not]: _constants.APPOINTMENT_STATUSES.CANCELLED
                            }
                        },
                        timeQueryWhereClause
                    ]
                },
                bind: timeQueryBindParams,
                transaction
            });
            if (conflictCount > 0) throw new _errors.EditConflictError();
            const location = await Location.findByPk(locationId, {
                transaction
            });
            if (!location) throw new _errors.NotFoundError('Location not found');
            await PatientFacility.findOrCreate({
                where: {
                    patientId,
                    facilityId: location.facilityId
                },
                transaction
            });
            const appointment = await Appointment.create(body, {
                transaction
            });
            if (procedureTypeIds) {
                await appointment.setProcedureTypes(procedureTypeIds);
            }
            if (email) {
                await sendAppointmentReminder({
                    appointmentId: appointment.id,
                    email,
                    facilityId: location.facilityId,
                    models,
                    settings,
                    transaction
                });
            }
            return appointment;
        });
        res.status(201).send(result);
    } catch (error) {
        res.status(error.status || 500).send();
    }
}));
appointments.put('/locationBooking/:id', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkPermission('write', 'Appointment');
    const { models, body, params, query } = req;
    const { id } = params;
    const { skipConflictCheck = false } = query;
    const { startTime, endTime, locationId, procedureTypeIds } = body;
    const { Appointment } = models;
    try {
        const result = await Appointment.sequelize.transaction(async (transaction)=>{
            const existingBooking = await Appointment.findByPk(id, {
                transaction
            });
            if (!existingBooking) {
                throw new _errors.NotFoundError();
            }
            if (!skipConflictCheck) {
                const [timeQueryWhereClause, timeQueryBindParams] = buildTimeQuery(startTime, endTime);
                const bookingTimeAlreadyTaken = await Appointment.findOne({
                    where: {
                        [_sequelize.Op.and]: [
                            {
                                id: {
                                    [_sequelize.Op.ne]: id
                                }
                            },
                            {
                                locationId
                            },
                            timeQueryWhereClause
                        ]
                    },
                    bind: timeQueryBindParams,
                    transaction
                });
                if (bookingTimeAlreadyTaken) {
                    throw new _errors.EditConflictError();
                }
            }
            const updatedRecord = await existingBooking.update(body, {
                transaction
            });
            await updatedRecord.setProcedureTypes(procedureTypeIds);
            return updatedRecord;
        });
        res.status(200).send(result);
    } catch (error) {
        res.status(error.status || 500).send();
    }
}));
const getAppointmentTypeWhereQuery = (type, facilityId)=>{
    const facilityIdField = type === 'outpatient' ? '$locationGroup.facility_id$' : '$location.facility_id$';
    if (type === 'outpatient') {
        return {
            locationGroupId: {
                [_sequelize.Op.not]: null
            },
            [facilityIdField]: facilityId
        };
    }
    return {
        locationId: {
            [_sequelize.Op.not]: null
        },
        [facilityIdField]: facilityId
    };
};
appointments.get('/hasPastAppointments/:patientId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkListOrReadPermission('Appointment');
    const { models, params, query } = req;
    const { patientId } = params;
    const { type, facilityId } = query;
    const [{ hasPastAppointment }] = await models.Appointment.sequelize.query(`
      SELECT EXISTS(
        SELECT 1 FROM appointments
        ${type === 'outpatient' ? 'LEFT JOIN location_groups ON appointments.location_group_id = location_groups.id' : 'LEFT JOIN locations ON appointments.location_id = locations.id'}
        WHERE patient_id = :patientId
          AND status != :cancelledStatus
          AND start_time < NOW()::date_time_string
          AND ${type === 'outpatient' ? 'location_groups.facility_id = :facilityId' : 'locations.facility_id = :facilityId'}
        LIMIT 1
      ) as "hasPastAppointment"
    `, {
        replacements: {
            patientId,
            facilityId,
            cancelledStatus: _constants.APPOINTMENT_STATUSES.CANCELLED,
            typeColumn: type === 'outpatient' ? 'location_group_id' : 'location_id'
        },
        type: _sequelize.QueryTypes.SELECT
    });
    res.send(hasPastAppointment);
}));
appointments.get('/upcomingAppointments/:patientId', (0, _expressasynchandler.default)(async (req, res)=>{
    req.checkListOrReadPermission('Appointment');
    const { models, params, query } = req;
    const { patientId } = params;
    const { type, facilityId, orderBy = 'startTime', order = 'ASC' } = query;
    const { Appointment } = models;
    const sortKeys = {
        startTime: 'startTime',
        outpatientAppointmentArea: [
            'locationGroup',
            'name'
        ],
        bookingArea: [
            'location',
            'locationGroup',
            'name'
        ],
        clinician: [
            'clinician',
            'displayName'
        ],
        appointmentType: [
            'appointmentType',
            'name'
        ],
        bookingType: [
            'bookingType',
            'name'
        ],
        location: [
            'location',
            'name'
        ]
    };
    const sortKey = sortKeys[orderBy] || 'startTime';
    const sortOrder = order.toUpperCase() === 'DESC' ? 'DESC' : 'ASC';
    // Build order clause - handle nested associations
    const orderClause = Array.isArray(sortKey) ? [
        sortKey.concat([
            sortOrder
        ])
    ] : [
        [
            sortKey,
            sortOrder
        ]
    ];
    const upcomingAppointments = await Appointment.findAll({
        where: {
            patientId,
            status: {
                [_sequelize.Op.not]: _constants.APPOINTMENT_STATUSES.CANCELLED
            },
            startTime: {
                [_sequelize.Op.gt]: new Date()
            },
            ...getAppointmentTypeWhereQuery(type, facilityId)
        },
        include: [
            'locationGroup',
            {
                association: 'location',
                include: [
                    'locationGroup'
                ]
            },
            'clinician',
            'appointmentType',
            'bookingType',
            'patient'
        ],
        order: orderClause
    });
    res.send(upcomingAppointments);
}));

//# sourceMappingURL=appointments.js.map